/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package kenya.eclipse.ast;

import java.util.ArrayList;
import java.util.List;

import kenya.eclipse.ast.bindings.Bindings;
import kenya.eclipse.multieditor.kenya.util.LocationUtils;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.ICheckedCode;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.AFuncDecDeclaration;
import minijava.node.Node;
import minijava.node.Start;
import minijava.node.TIdentifier;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;

/**
 * this class serves to find occurrences of a particular element in Kenya
 * source code. To be able to utilise this functionality, the
 * AST (Start), ICheckedCode, document and selected location must be known.
 * The finder is first initialised with the above data, after which a single
 * call to perform() will yield the result.
 * For each lookup, the Finder MUST be reinitialised, otherwise unexpected
 * results may appear.
 * The resulting List contains instances of TIdentifiers that are directly
 * related to the Node that is to be looked up. This will be the
 * list of TIdentifiers that are occurrences of the same instance in the code.
 * 
 * @author Thomas Timbul
 */
public class OccurrencesFinder extends DepthFirstAdapter {
	
	public static final String IS_WRITEACCESS= "writeAccess"; //$NON-NLS-1$
	public static final String IS_VARIABLE= "variable"; //$NON-NLS-1$
	
	private Start fRoot;
	private ICheckedCode fCode;
	private TIdentifier fSelectedNode;
	private IBinding fTarget;
	private List fUsages;
	
	public OccurrencesFinder(IBinding target) {
		this();
		fTarget= target;
	}
	
	public OccurrencesFinder() {
		super();
	}
	
	public String initialize(Start root, ICheckedCode code, IDocument doc, ISourceCodeLocation loc) {
		Position pos = LocationUtils.convert(loc, doc);
		return initialize(root, code, doc, pos.getOffset(), pos.getLength());
	}
	
	public String initialize(Start root, ICheckedCode code, IDocument doc, int offset, int length) {
		Node node = NodeFinder.perform(root, doc, new Position(offset, length));
		
		if (!(node instanceof TIdentifier))
			return "Cannot search for the current selection. Please select a valid Kenya element name.";
		fCode = code;
		fRoot = root;
		fSelectedNode= (TIdentifier)node;
		fTarget= Resolver.resolve(fSelectedNode, fCode);
		if (fTarget == null)
			return "Selected Kenya element is unknown.";
		
		fUsages = new ArrayList();
		return null;
	}
	
	public IBinding getTarget() {
		return fTarget;
	}
	
	public String getExpandedSelectionText() {
		return NodeToString.toString(fSelectedNode);
	}
	
	public List perform() {
		fRoot.apply(this);
		return fUsages;
	}
	
	public void caseTIdentifier(TIdentifier node) {
		match(node, fUsages, Resolver.resolve(node, fCode));
	}
	/* (non-Javadoc)
	 * @see minijava.analysis.DepthFirstAdapter#caseAFuncDecDeclaration(minijava.node.AFuncDecDeclaration)
	 */
	public void caseAFuncDecDeclaration(AFuncDecDeclaration node) {
		super.caseAFuncDecDeclaration(node);
	}
	private boolean match(TIdentifier node, List result, IBinding binding) {
		if (binding != null && Bindings.equals(binding, fTarget)) {
			result.add(node);
			return true;
		}
		return false;
	}
}
