/*
 * Created on 18-Nov-2004
 */
package kenya.eclipse.buildext.util;

import java.io.IOException;
import java.io.Reader;

import kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import kenya.sourceCodeInformation.interfaces.ISourceCodeInformation;
import kenya.sourceCodeInformation.interfaces.ISourceCodeWarning;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.text.Position;

/**
 * provides facilities to set the attributes on IMarker objects.
 * The IMarker and an ISouceCodeInformation is supplied in order
 * to set all attributes on the marker according to the Information.
 * This includes the location as well as severity of the marker. It
 * is therefore assumed that the IMarker supports the attributes
 * <ul>
 *   <li>IMarker.MESSAGE
 *   <li>IMarker.SEVERITY
 *   <li>IMarker.LINE_NUMBER
 *   <li>IMarker.CHAR_START
 *   <li>IMarker.CHAR_END
 *   <li>IMarker.USER_EDITABLE
 * </ul>
 * If this is not the case, no Exception will be thrown nor will there be
 * any user notification. The attributes will simply not be set, however
 * there is no guarantee whether any, or which attributes may have been
 * set in this event.
 * 
 * @author Thomas Timbul
 */
public class MarkerPropertySetter {
	
	/**
	 * sets the attributes of an IMarker according to an ISourceCodeInformation
	 * including the line number. The exact position (see char_start and char_end)
	 * has to be set separately using setLocation.
	 * 
	 * @param marker the IMarker whose attributes to set
	 * @param info the Information to use for setting properties
	 */
	public static void setProps(IMarker marker, ISourceCodeInformation info) {
		try {
			marker.setAttribute(IMarker.MESSAGE, read(info.getHumanStyleMessage()));
			/*
			 * set the severity...
			 * NOTE:
			 * the instanceof stuff is a workaround because
			 * the overloaded methods were not being called
			 * properly as the Runtime type
			 * was set to ISourceCodeInformation by casting
			 * when iterating through the list of infos
			 */
			if(info instanceof ISourceCodeWarning) {
				marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_WARNING);
			} else if(info instanceof ISourceCodeError) {
				marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
			} else {
				marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_INFO);
			}
			
			marker.setAttribute(IMarker.LINE_NUMBER, info.getLocation().getLineNumber());
			
			marker.setAttribute(IMarker.USER_EDITABLE, false);
			
		} catch(CoreException ex) {
			//what happened?
		}
	}
	
	/**
	 * returns the contents of a Reader as a String
	 * @param r the Reader to read
	 * @return a String representing the contents of the Reader
	 */
	private static String read(Reader r) {
		StringBuffer b = new StringBuffer();
		try {
			char c;
			while(true) {
				c = (char)r.read();
				if(!Character.isDefined(c)) {
					break;
				}
				b.append(c);
			}
		} catch(IOException e) {
			//ignore
		}
		return b.toString();
	}
	
	/**
	 * sets the char_start and char_end attributes of an IMarker
	 * according to a Position. If pos is null, both attributed are set to -1.
	 * @param marker the IMarker whose location to set
	 * @param loc the location which provides the information
	 * @throws CoreException if the marker does not exist or another error occurred
	 */
	public static void setLocation(IMarker marker, Position pos)
			throws CoreException {
		
		if(pos==null) {
			pos = new Position(0, 0);
		}
		
		marker.setAttribute(IMarker.CHAR_START, pos.getOffset());
		marker.setAttribute(IMarker.CHAR_END, pos.getOffset()+pos.getLength());
	}
	
}
