/*
 * Created on 17-Dec-2004
 */
package kenya.eclipse.debug;

import java.util.ArrayList;

import kenya.eclipse.KenyaConstants;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;

/**
 * Provides various methods to search for Kenya files in the workspace.
 * @author Thomas Timbul
 */
public class KenyaTypeSearchEngine {
	
	public IPath[] search(IResource[] ress) throws CoreException {
		ArrayList r = new ArrayList();
		for(int i=0; i<ress.length; i++) {
			r.addAll(recursiveSearch(ress[i]));
		}
		return (IPath[])r.toArray(new IPath[0]);
	}
	
	public IPath[] search(IResource res) throws CoreException {
		return (IPath[])recursiveSearch(res).toArray(new IPath[0]);
	}
	
	/**
	 * returns the IPaths of all Kenya files (file ending .k) in
	 * the given project, relative to the workspace root.
	 * @param root
	 * @return
	 * @throws CoreException
	 */
	public IPath[] search(IContainer root) throws CoreException {
		ArrayList paths = recursiveSearch(root);
		
		return (IPath[])paths.toArray(new IPath[paths.size()]);
	}
	
	/**
	 * Returns an array with exactly one element if <code>file</code> is
	 * a Kenya file (file ending .k).<br>
	 * The element in the array is the Path of <code>file</code> relative
	 * to the workspace root obtained by calling
	 * <code>file.getFullPath()</code>.<br>
	 * Otherwise, the returned value is <code>new IPath[0]</code>.
	 * 
	 * @param file the file to search
	 * @return an array of IPaths containing the path of file if file is a Kenya file,
	 * or an empty array if this is not the case.
	 * @throws CoreException no exception is thrown, the throws clause is to
	 * overload the search method properly
	 */
	public IPath[] search(IFile file) throws CoreException {
		return isKenyaFile(file)
		         ?new IPath[] {file.getFullPath()}
		         :new IPath[0];
	}
	
	protected boolean isKenyaFile(IFile file) {
		return KenyaConstants.EXTENSION_k.equals(file.getFileExtension());
	}
	
	protected ArrayList recursiveSearch(IResource res) throws CoreException {
		ArrayList r = new ArrayList();
		
		if (res instanceof IContainer) {
			return recursiveSearch((IContainer)res);
		} else if (res instanceof IFile) {
			IPath[] p = search((IFile)res);
			if (p.length == 1) {
				r.add(p[0]);
			}
		}
		
		return r;
	}
	
	protected ArrayList recursiveSearch(IContainer root) throws CoreException {
		ArrayList paths = new ArrayList();
		
		if(root instanceof IProject) {
			IProject r = (IProject)root;
			if(!r.hasNature(KenyaConstants.K_NATURE_ID)) {
				return paths;
			}
		}
		
		IResource[] r = root.members();
		for(int i = 0; i < r.length; i++) {
			if(r[i] instanceof IContainer) {
				paths.addAll(recursiveSearch((IContainer)r[i]));
			} else if(r[i] instanceof IFile) {
				IFile f = (IFile)r[i];
				if(isKenyaFile(f)) {
					paths.add(f.getFullPath());
				}
			}
		}
		
		return paths;
	}
	
}
