/*
 * Created on 13-Dec-2004
 */
package kenya.eclipse.debug;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.Comparator;

import kenya.eclipse.KenyaPlugin;
import mediator.IJavaCode;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;

/**
 * virtual Process to encapsulate IJavaCode.execute
 * @author Thomas Timbul
 */
public class VKRunProcess extends AbstractVKProcess {
	
	// * in * //
	private PipedInputStream stdInPIS;
	private PipedOutputStream stdInPOS;
	private BufferedReader stdInReader;
	private PrintWriter stdInWriter;
  
  // * out * //
	private PipedInputStream stdOutPIS;
	private PipedOutputStream stdOutPOS;
	private BufferedWriter stdOutWriter;
  
  // * err * //
	private PipedInputStream stdErrPIS;
	private BufferedWriter stdErrWriter;
  
	private IJavaCode code;
	private String[] args;
	
	private VKRunProcessStreamManager fStreamManager;
	
	/**
	 * Creates and executes a new VKProcess with given parameters
	 * 
	 * @param workingDirectory workingDirectory for this process
	 * @param code the IJavaCode to execute
	 * @param className the name under which to execute the code
	 * @param args process arguments (program arguments supplied to the process)
	 */
  public VKRunProcess(final String workingDirectory, final IJavaCode code, final String className, final String[] args) {
		super(workingDirectory, className);
  	if(code==null) {
			finish(BAD_CODE);
			return;
		}
  	this.code = code;
  	this.args = args!=null?args:new String[0];
		
		if(hasFinished()) {
			return;
		}
		
		try {
			//construct that uber complicated stream stuff
			stdInPOS = new PipedOutputStream();
			stdInPIS = new PipedInputStream(stdInPOS);
			stdInReader = new BufferedReader(new InputStreamReader(stdInPIS));
			stdInWriter = new PrintWriter(stdInPOS);
			stdOutPIS = new PipedInputStream();
			stdOutPOS = new PipedOutputStream(stdOutPIS);
		  stdOutWriter = new BufferedWriter(new OutputStreamWriter(stdOutPOS));
		  stdErrPIS = new PipedInputStream();
		  stdErrWriter = new BufferedWriter(new OutputStreamWriter(new PipedOutputStream(stdErrPIS)));
		  
		} catch(IOException e) {
			//hope this never happens
			//e.printStackTrace();
			KenyaPlugin.log("Launch of "+className+" failed:\n" +
					"Could not create piped streams.", e);
			finish(STREAM_CREATION_FAILURE);
		}
		//if all is well...
		
		//check that we can execute
		if(!hasFinished()) {
			//then do it
			initProcess();
		}
	}
	
  protected void initProcess() {
  	fJob = new Job(VK_PROCESS_NAME) {
			public IStatus run(IProgressMonitor monitor) {
				
				//TODO do not compile to working directory, have a build directory instead
				File[] snapshot0 = workingDirectory.listFiles();
				
				code.execute(workingDirectory.getAbsolutePath(), className, stdInReader, new BufferedWriter(stdInWriter), stdOutWriter, stdErrWriter, args);
				
				//get rid of class and java files that we created
				File[] snapshot1 = workingDirectory.listFiles();
				
				for(int i=0; i<snapshot1.length; i++) {
					File file = snapshot1[i];
					int r = Arrays.binarySearch(snapshot0, file, new FileNameComparator());
					if(r<0) {
						//not found - if it is java or class, then delete it
						String name = file.getName();
						int dotIndex = name.lastIndexOf('.');
						if(dotIndex>0) {
							String ending = name.substring(dotIndex);
							
							//note: constraining on java and class files, anything else may
							//be purposefully created in the directory.
							//Maybe we should be very careful with the deleteOnExit()...
							
							if("java".equals(ending) || "class".equals(ending)) {
								if(!file.delete()) {
									//file.deleteOnExit(); //careful, maybe its better to leave it be
								}
							}
						}
						
					}
				}
				
				//finally, terminate with exit value 0 (success)
				finish(0);
				return Status.OK_STATUS;
			}
		};
		fJob.setSystem(true);
		fJob.schedule();
  }
  
	/**
	 * does a clean up with respect to all the streams declared above
	 * i.e. closes the writers
	 */
	protected void dispose() {
		
		//close all the writers.
		if(stdOutWriter!=null) {
			try {stdOutWriter.close();} catch(IOException e) {/*who cares*/}
//				stdOutWriter = null;
		}
		if(stdErrWriter!=null) {
			try {stdErrWriter.close();} catch(IOException e) {/*who cares*/}
//				stdErrWriter = null;
		}
		if(stdInWriter!=null) {
			stdInWriter.close();
//				stdInWriter = null;
		}
		
		//super.dispose();
		
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Process#destroy()
	 */
	public void destroy() {
		if(!hasFinished()) {
			code.killLastProcess();
			finish(PREMATURE_DESTRUCTION);
			dispose(); //people dont read streams when they tell you to destroy yourself
		}
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Process#getErrorStream()
	 */
	public InputStream getErrorStream() {
		return new BufferedInputStream(stdErrPIS);
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Process#getInputStream()
	 */
	public InputStream getInputStream() {
		return new BufferedInputStream(stdOutPIS);
		
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Process#getOutputStream()
	 */
	public OutputStream getOutputStream() {
		return new BufferedOutputStream(stdInPOS);

	}
	
	
	class FileNameComparator implements Comparator {
		/* (non-Javadoc)
		 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
		 */
		public int compare(Object o1, Object o2) {
			return 0; //in this context they are equal
		}
		public int compare(File f1, File f2) {
			if(f1.isDirectory()) {
				if(f2.isDirectory()) {
					return f1.getName().compareTo(f2.getName());
				} else {
					return 1;
				}
			} else if(f1.isFile()) {
				if(f2.isDirectory()) {
					return -1;
				} else if(f2.isFile()) {
					return f1.getName().compareTo(f2.getName());
				} else { //not existent
					return 1;
				}
			} else { //not existent
				return -1;
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.AbstractVKProcess#getStreamManager()
	 */
	protected IProcessStreamManager getStreamManager() {
		if (fStreamManager == null) {
			fStreamManager = new VKRunProcessStreamManager();
		}
		return fStreamManager;
	}
	
	class VKRunProcessStreamManager implements IProcessStreamManager {
		
		public void closeInput() throws IOException {
			if(stdInWriter!=null) {
				stdInWriter.close();
			}
		}
		
		public void closeOutput() throws IOException {
			if(stdOutWriter!=null) {
				stdOutWriter.close();
			}
		}
		
		public void closeErr() throws IOException {
			if(stdErrWriter!=null) {
				stdErrWriter.close();
			}
		}
	}
}
