/*
 * Created on 25-Dec-2004
 */
package kenya.eclipse.debug.bridge;

import java.lang.reflect.InvocationTargetException;

/**
 * <p>
 * Describes an agent that can communicate with a remote object through
 * an ICommunicationBridge. The advantage of this method over reflection is that
 * the target can restrict method invocation in its implementation. The target
 * may choose to call reflection methods to implement.
 * </p>
 * <p>The disadvantage compared to reflection is that both source and target have
 * to be registered with the Bridge in order to be able to communicate.
 * </p>
 * <p>Implementors can choose whether to allow multiple bridges or not. In the
 * former case setBridge should add to a collection of bridges, whereas in the
 * latter it would simply set a variable.
 * </p>
 * @author Thomas Timbul
 */
public interface IBridgeCommunicator {
	
	
	/**
	 * set the bridge
	 * Implementors should call <code>bridge.addPort(this);</code> in this method
	 * @param bridge the bridge to use for communication
	 */
	public void setBridge(ICommunicationBridge bridge);
	
	/**
	 * removes the bridge from this communicator and makes
	 * it unavailable for calls in the context of this
	 * @param bridge the bridge to remove
	 */
	public void removeBridge(ICommunicationBridge bridge);
	
	/**
	 * 
	 * @return the bridge
	 */
	public ICommunicationBridge getBridge();
	
	/**
	 * invokes method with given name and arguments in this Communicator.
	 * Implementers may restrict on any method and should throw an
	 * InvocationTarget Exception wrapping around another exception that explicitly
	 * describes the cause of the failure, such as inexistent method, a restriction
	 * as to which methods may be called remotely, etc.
	 * If the remote method has return type void, <code>null</code> is to be returned.
	 * 
	 * @param methName name of the method to invoke
	 * @param args arguments for the method call
	 * @return result of method call or <code>null</code> if the method has type <code>void</code>
	 * @throws InvocationTargetException if the method does not exist
	 */
	public Object invokeRemote(String methName, Object[] args) throws InvocationTargetException;
	
}
