/*
 * Created on 22-Dec-2004
 */
package kenya.eclipse.debug.model;

import java.util.HashMap;
import java.util.Map.Entry;

import kenya.eclipse.debug.launcher.IKenyaLaunchConfigurationConstants;
import kenya.interpreter.util.InterpreterInspectorContentProvider;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IRegisterGroup;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;
import org.eclipse.debug.core.model.IVariable;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.texteditor.DocumentProviderRegistry;
import org.wellquite.kenya.stackMachine.IPointListener;
import org.wellquite.kenya.stackMachine.scope.IMethodScope;

/**
 * @author Thomas Timbul
 */
public class KStackFrame extends KenyaDebugElement implements IStackFrame, IPointListener {
	
	private IThread fThread;
	
	private IMethodScope fScope;
	
	//used to inspect the contents of the stack machine (which we dont need itself)
	private InterpreterInspectorContentProvider inspector
	  = new InterpreterInspectorContentProvider();
	
	//holds the location of execution point in this frame
	private volatile ISourceCodeLocation fLoc;
	private volatile int offset = 0;
	private boolean fUpdateLocation;
	
	//holds the name of the stackframe
	private String fName;
	
	private HashMap fVarMap = new HashMap();
	
	KStackFrame(KenyaDebugTarget target, IThread thread, IMethodScope scope, String name, ISourceCodeLocation loc) {
		this(target, thread, scope, name, loc, true);
	}
	
	KStackFrame(KenyaDebugTarget target, IThread thread, IMethodScope scope, String name, ISourceCodeLocation loc, boolean updateLocation) {
		super(target);
		fThread = thread;
		fScope = scope;
		fName = name;
		fLoc = loc;
		fUpdateLocation = updateLocation;
		
		reCalculateOffset();
	}
	
	private void reCalculateOffset() {
		//getCharStart is expected to return the absolute offset in the
		//source document. This means we have to convert and add the
		//line number information
		
		try {
			String sourcePath = getLaunch().getLaunchConfiguration().getAttribute(IKenyaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME, (String)null);
			IEditorInput e = KenyaDebugModel.getModelPresentation().getEditorInput(sourcePath);
			IDocument d = DocumentProviderRegistry.getDefault().getDocumentProvider(e).getDocument(e);
			if(d==null) return;
			offset = d.getLineOffset(getLineNumber());
			return;
			
		} catch(NullPointerException e) {
		} catch(DebugException e) {
		} catch(BadLocationException e) {
		} catch(CoreException e) {
		}
		offset = -1;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStackFrame#getThread()
	 */
	public IThread getThread() {
		return fThread;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStackFrame#getVariables()
	 */
	public IVariable[] getVariables() throws DebugException {
		Object[] o = inspector.getChildren(fScope);
		KVariable[] v = new KVariable[o.length];
		
		for(int i = 0; i < v.length; i++) {
			Object key = (Entry)o[i];
			
			KVariable var = (KVariable)fVarMap.get(key);
			if(var==null) {
				var = new KVariable(fTarget, (Entry)o[i]);
				fVarMap.put(key, var);
			}
			v[i] = var;
			
		}
		
		return v;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStackFrame#hasVariables()
	 */
	public boolean hasVariables() throws DebugException {
		return inspector.hasChildren(fScope);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStackFrame#getLineNumber()
	 */
	public synchronized int getLineNumber() throws DebugException {
		return fLoc==null?-1:fLoc.getLineNumber()-1;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStackFrame#getCharStart()
	 */
	public synchronized int getCharStart() throws DebugException {
		return fLoc==null?-1:(offset<0)?-1:offset + fLoc.getColumnNumber() - 1;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStackFrame#getCharEnd()
	 */
	public synchronized int getCharEnd() throws DebugException {
		return fLoc==null?-1:getCharStart() + fLoc.getTokenLength();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStackFrame#getName()
	 */
	public String getName() throws DebugException {
		return fName;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStackFrame#getRegisterGroups()
	 */
	public IRegisterGroup[] getRegisterGroups() throws DebugException {
		return new IRegisterGroup[0];
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStackFrame#hasRegisterGroups()
	 */
	public boolean hasRegisterGroups() throws DebugException {
		return false;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#canStepInto()
	 */
	public boolean canStepInto() {
		return isSuspended() && isStepping();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#canStepOver()
	 */
	public boolean canStepOver() {
		return canStepInto();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#canStepReturn()
	 */
	public boolean canStepReturn() {
		return canStepInto();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#isStepping()
	 */
	public boolean isStepping() {
		return getThread().isStepping();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#stepInto()
	 */
	public void stepInto() throws DebugException {
		fireResumeEvent(DebugEvent.STEP_INTO);
		getThread().stepInto();
		fireSuspendEvent(DebugEvent.STEP_END);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#stepOver()
	 */
	public void stepOver() throws DebugException {
		fireResumeEvent(DebugEvent.STEP_OVER);
		getThread().stepOver();
		fireSuspendEvent(DebugEvent.STEP_END);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#stepReturn()
	 */
	public void stepReturn() throws DebugException {
		fireResumeEvent(DebugEvent.STEP_RETURN);
		getThread().stepReturn();
		fireSuspendEvent(DebugEvent.STEP_END);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#canResume()
	 */
	public boolean canResume() {
		return getThread().canResume();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#canSuspend()
	 */
	public boolean canSuspend() {
		return getThread().canSuspend();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#isSuspended()
	 */
	public boolean isSuspended() {
		return getThread().isSuspended();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#resume()
	 */
	public void resume() throws DebugException {
		fireResumeEvent(DebugEvent.EVALUATION);
		getThread().resume();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#suspend()
	 */
	public void suspend() throws DebugException {
		getThread().suspend();
		fireSuspendEvent(DebugEvent.CLIENT_REQUEST);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#canTerminate()
	 */
	public boolean canTerminate() {
		return getThread().canTerminate();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#isTerminated()
	 */
	public boolean isTerminated() {
		return getThread().isTerminated();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#terminate()
	 */
	public void terminate() throws DebugException {
		getThread().terminate();
		fireTerminateEvent();
	}
	
	/* (non-Javadoc)
	 * @see org.wellquite.kenya.stackMachine.IPointListener#pointReached(java.lang.Object)
	 */
	public synchronized void pointReached(Object data) {
		if (data instanceof ISourceCodeLocation) {
//			int col = fLoc.getColumnNumber();
			if(!fUpdateLocation) return; //don't update if this flag is set
			fLoc = (ISourceCodeLocation)data;
//			int colDiff = fLoc.getColumnNumber() - col;
//			
//			offset+=colDiff;
			reCalculateOffset();
		} else {
			System.out.println("pointReached data: "+data.getClass().getName());
		}
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object obj) {
		return (obj instanceof KStackFrame && equals((KStackFrame)obj))
		    || super.equals(obj);
	}
	
	public boolean equals(KStackFrame obj) {
		try {
			return obj.fTarget==fTarget
			    && obj.fThread==fThread
			    && obj.fScope==fScope
					&& obj.getVariables().equals(getVariables());
		} catch(DebugException e) {
			return false;
		}
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		return fScope.hashCode();
	}
	
	/**
	 * updates the internal state of all variables if required
	 *
	 */
	public synchronized void updateBeforeResume() {
//		try {
//			if(hasVariables()) {
//				KVariable[] vars = (KVariable[])getVariables();
//				for(int i=0; i<vars.length; i++) {
//					vars[i].updateBeforeResume();
//				}
//			}
//		} catch(DebugException e) {
//		}
	}
	
}