/*
 * Created on 22-Dec-2004
 */
package kenya.eclipse.debug.model;

import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.debug.model.breakpoints.KenyaLineBreakpoint;
import kenya.interpreter.util.InterpreterInspectorListContentProvider;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.ICheckedCode;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IBreakpointManager;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.ui.IDebugModelPresentation;

/**
 * contains utilities for debugging Kenya in the Eclipse debugging model
 * @author Thomas Timbul
 */
public class KenyaDebugModel {
	
	public static final String MODEL_ID = KenyaPlugin.getPluginId();
	
	private static InterpreterInspectorListContentProvider inspector;
	
	private static IDebugModelPresentation fPresentation;
	
	/**
	 * Not to be instantiated.
	 */
	private KenyaDebugModel() {
	}
	
	public static String getModelIdentifier() {
		return MODEL_ID;
	}
	
	public static IDebugModelPresentation getModelPresentation() {
		if (fPresentation == null) {
			fPresentation = new KenyaDebugModelPresentation();
		}
		return fPresentation;
	}
	
	/**
	 * creates a new VKDebugProcess with the given arguments
	 * @param workingDirectory
	 * @param checkedCode
	 * @param arguments
	 * @return
	 */
	public static Process newDebugProcess(ILaunch launch, String workingDirectory, ICheckedCode checkedCode, String[] arguments, boolean stopInMain) {
		return new VKDebugProcess(launch, workingDirectory, checkedCode, arguments, stopInMain);
	}
	
	/**
	 * for instance reuse, call this method to get an instance of the
	 * InterpreterInspectorListContentProvider.
	 * @return InterpreterInspectorListContentProvider instance
	 */
	public static InterpreterInspectorListContentProvider getInspectorListContentProvider() {
		if(inspector==null) {
			inspector = new InterpreterInspectorListContentProvider();
		}
		return inspector;
		
	}
	
	public static KenyaLineBreakpoint lineBreakpointExists(IResource resource, int lineNumber) throws CoreException {
		String modelId= getModelIdentifier();
		String markerType= KenyaLineBreakpoint.getMarkerType();
		IBreakpointManager manager= DebugPlugin.getDefault().getBreakpointManager();
		IBreakpoint[] breakpoints= manager.getBreakpoints(modelId);
		for (int i = 0; i < breakpoints.length; i++) {
			if (!(breakpoints[i] instanceof KenyaLineBreakpoint)) {
				continue;
			}
			KenyaLineBreakpoint breakpoint = (KenyaLineBreakpoint) breakpoints[i];
			IMarker marker = breakpoint.getMarker();
			if (marker != null && marker.exists() && marker.getType().equals(markerType)) {
				if(marker.getResource().equals(resource)) {
					if (breakpoint.getLineNumber() == lineNumber) {
						return breakpoint;
					}
				}
			}
		}
		return null;
	}
	
	/**
	 * Creates and returns a line breakpoint in the given resource,
	 * at the given line number. The marker associated with the
	 * breakpoint will be created on the same resource. If a character
	 * range within the line is known, it may be specified by charStart/charEnd.
	 * If hitCount is > 0, the breakpoint will suspend execution when it is
	 * "hit" the specified number of times.
	 * 
	 * @param resource the resource on which to create the associated breakpoint
	 *  marker
	 * @param lineNumber the lineNumber on which the breakpoint is set - line
	 *   numbers are 1 based, associated with the source file in which
	 *   the breakpoint is set
	 * @param charStart the first character index associated with the breakpoint,
	 *   or -1 if unspecified, in the source file in which the breakpoint is set
 	 * @param charEnd the last character index associated with the breakpoint,
	 *   or -1 if unspecified, in the source file in which the breakpoint is set
	 * @param register whether to add this breakpoint to the breakpoint manager
	 * @return a line breakpoint
	 * @exception CoreException If this method fails. Reasons include:<ul> 
	 *<li>Failure creating underlying marker.  The exception's status contains
	 * the underlying exception responsible for the failure.</li></ul>
	 */
	public static KenyaLineBreakpoint createLineBreakpoint(IResource resource, ISourceCodeLocation location, int start, int end, boolean register) throws CoreException {
		KenyaLineBreakpoint existing = lineBreakpointExists(resource, location.getLineNumber());
		return existing!=null?existing:new KenyaLineBreakpoint(resource, location, start, end, register);
	}
	
}
