/*
 * Created on 13-Dec-2004
 */
package kenya.eclipse.debug.model;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.lang.reflect.InvocationTargetException;

import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.debug.AbstractVKProcess;
import kenya.eclipse.debug.IProcessStreamManager;
import kenya.eclipse.debug.KenyaRemote;
import kenya.eclipse.debug.bridge.ICommunicationBridge;
import kenya.interpreter.util.InterpreterLastPointReachedCatcher;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.ICheckedCode;
import mediator.stackMachine.IStackMachineInformationProvider;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.debug.core.ILaunch;
import org.wellquite.kenya.stackMachine.ClosureMiscHelper;
import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.scope.ClosureScope;
import org.wellquite.kenya.stackMachine.scope.IClosureScope;
import org.wellquite.kenya.stackMachine.types.ArrayTypeFactory;
import org.wellquite.kenya.stackMachine.types.StringTypeFactory;
import org.wellquite.kenya.stackMachine.types.interfaces.IAtomicClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;
import org.wellquite.kenya.stackMachine.types.interfaces.IStringType;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * virtual Process to encapsulate ICheckedCode.execute
 * @author Thomas Timbul
 */
public class VKDebugProcess extends AbstractVKProcess implements KenyaRemote {
	
	// * in * //
	private PipedInputStream stdInPIS;
	private PipedOutputStream stdInPOS;
	private InputStream inStream;
	private BufferedOutputStream bufferedIn;
	
	// * out * //
	private PipedInputStream stdOutPIS;
	private PipedOutputStream stdOutPOS;
	private PrintStream outStream;
	private BufferedInputStream bufferedOut;
	
	// * err * //
	private PipedInputStream stdErrPIS;
	private PipedOutputStream stdErrPOS;
	private PrintStream errStream;
	private BufferedInputStream bufferedErr;
	
	private StackMachine sm;
	private IStackMachineInformationProvider ismip;
	
	private ICommunicationBridge fBridge;
	
	private VKDebugProcessStreamManager fStreamManager;
	
	/**
	 * Creates and executes a new VKProcess with given parameters
	 * 
	 * @param launch the ILaunch that was used to launch this process
	 * @param workingDirectory workingDirectory for this process
	 * @param code the ICheckedCode to execute
	 * @param className the name under which to execute the code
	 * @param args process arguments (program arguments supplied to the process)
	 * @param whether or not the process should stop after initialisation
	 */
	VKDebugProcess(
			final ILaunch launch,
			final String workingDirectory,
			final ICheckedCode code,
			final String[] args,
			final boolean stopInMain) {
		super(workingDirectory, code.getBaseStackMachine().getEntryPointClass().getName());
		
		if(code==null || code.isErroredCode()) {
			finish(BAD_CODE);
			return;
		}
		
		if(hasFinished()) {
			return;
		}
		
		//create and prepare stack machine
		getStackMachine();
		ismip = code.getBaseStackMachine();
		
		//stack machine setup
		try {
			IClosureScope scope = new ClosureScope();
			ClosureMiscHelper.executeClosureInScope(scope,
					ismip.getPreInitClosure(), sm);
		} catch (Throwable t) {
			KenyaPlugin.log(t);
		}
		
		
		try {
			//construct that uber complicated stream stuff
			stdInPOS = new PipedOutputStream();
			stdInPIS = new PipedInputStream(stdInPOS);
			inStream = new BufferedInputStream(stdInPIS);
			bufferedIn = new BufferedOutputStream(stdInPOS);
			
			stdOutPIS = new PipedInputStream();
			stdOutPOS = new PipedOutputStream(stdOutPIS);
			outStream = new PrintStream(stdOutPOS, true);
			bufferedOut = new BufferedInputStream(stdOutPIS);
			
			stdErrPIS = new PipedInputStream();
			stdErrPOS = new PipedOutputStream(stdErrPIS);
			errStream = new PrintStream(stdErrPOS, true);
			bufferedErr = new BufferedInputStream(stdErrPIS);
			
			sm.setOut(outStream);
			sm.setErr(errStream);
			sm.setIn(inStream);
			
		} catch(IOException e) {
			//hope this never happens
			//e.printStackTrace();
			KenyaPlugin.log("Launch of "+className+" failed:\n" +
					"Could not create piped streams.", e);
			finish(STREAM_CREATION_FAILURE);
		}
		//if all is well...
		
		//check that we can execute
		if(!hasFinished()) {
			
			//then do it
			fJob = new Job(VK_PROCESS_NAME) {
				
				public IStatus run(IProgressMonitor monitor) {
					try {
						if(monitor==null) {
							monitor = new NullProgressMonitor();
						}
						monitor.beginTask("run", 1);
						
						IInterpretedClass launchClass = ismip.getEntryPointClass();
						String launchMethod = ismip.getEntryPoint();
						
						InterpreterLastPointReachedCatcher ilprc = new InterpreterLastPointReachedCatcher(sm);
						sm.addPositionReachedListener(ilprc);
						
						if(monitor.isCanceled()) {
							finish(PREMATURE_DESTRUCTION);
							return Status.CANCEL_STATUS;
						}
						try {
							if (ismip.entryPointTakesArguments()) {
								String[] argsString = args;
								
								IType[] argsIType = new IType[argsString.length];
								for (int idx = 0; idx < argsString.length; idx++) {
									argsIType[idx] = StringTypeFactory
									.createStringType(argsString[idx]);
								}
								sm.push(ArrayTypeFactory.createArrayType(argsIType, IStringType.TYPE_NAME));
							}
							
							sm.setStepMode(stopInMain); //stop if so requested
							sm.setStepMode(true);
							
							sm.invokeMethod(launchClass, launchMethod);
							
						} catch (Throwable t) {
							//this is part of stackmachine execution, dont throw outside!!
							//ie, this error is shown to the user on the console as part of program output
							try {
								sm.shutdown();
							} catch(Exception ex) {
							}
							IAtomicClosure lastClosure = ilprc.getLastClosure();
							ISourceCodeLocation loc = (ISourceCodeLocation) ilprc.getLastPointData();
//							String message = t.toString();
							if (loc == null || lastClosure == null) {
								errStream.println(t);
							} else {
								errStream.println(t + " at "
										+ " line " + loc.getLineNumber()
										+ ", column " + loc.getColumnNumber());
							}
							finish(SUCCESS_BUT_PROGRAM_ERROR);
						}
					} catch (Throwable e) {
						KenyaPlugin.log(e);
						finish(INTERNAL_ERROR);
						return new Status(Status.ERROR, KenyaPlugin.getPluginId(), INTERNAL_ERROR, "", e);
					}
					
					monitor.worked(1);
					
					//finally, terminate with 'success'
					finish(SUCCESS);
					return Status.OK_STATUS;
				}
			};
			fJob.setSystem(true);
			fJob.schedule();
			
		}
	}
	
	/**
	 * does a clean up with respect to all the streams declared above
	 * i.e. closes the streams
	 */
	protected void dispose() {
		
		if(outStream!=null) {
			outStream.close();
		}
		if(errStream!=null) {
			errStream.close();
		}
		
		try {
			inStream.close();
		} catch(IOException e) {
			//ignore
		} catch(NullPointerException e) {
			//if inStream is null - ignore
		}
		
		if(sm!=null) {
			sm.shutdown();
			sm = null;
		}
		
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Process#destroy()
	 */
	public void destroy() {
		if(!hasFinished()) {
			if(sm!=null) {
				sm.shutdown();
			}
			finish(PREMATURE_DESTRUCTION);
		}
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Process#getErrorStream()
	 */
	public InputStream getErrorStream() {
		return bufferedErr;
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Process#getInputStream()
	 */
	public InputStream getInputStream() {
		return bufferedOut;
		
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Process#getOutputStream()
	 */
	public OutputStream getOutputStream() {
		return bufferedIn;
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.KenyaRemote#getStackMachine()
	 */
	public StackMachine getStackMachine() {
		if(sm==null) {
			sm = new StackMachine();
		}
		return sm;
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.KenyaRemote#getStackMachineInformationProvider()
	 */
	public IStackMachineInformationProvider getStackMachineInformationProvider() {
		return ismip;
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.bridge.IBridgeCommunicator#setBridge(kenya.eclipse.debug.bridge.ICommunicationBridge)
	 */
	public void setBridge(ICommunicationBridge bridge) {
		fBridge = bridge;
		fBridge.addPort(this);
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.bridge.IBridgeCommunicator#removeBridge(kenya.eclipse.debug.bridge.ICommunicationBridge)
	 */
	public void removeBridge(ICommunicationBridge bridge) {
		if(bridge==fBridge) {
			fBridge = null;
		}
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.bridge.IBridgeCommunicator#getBridge()
	 */
	public ICommunicationBridge getBridge() {
		return fBridge;
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.bridge.IBridgeCommunicator#invokeRemote(java.lang.String, java.lang.Object[])
	 */
	public Object invokeRemote(String methName, Object[] args) throws InvocationTargetException {
		if(GET_STACK_MACHINE.equals(methName)) {
			return getStackMachine();
		} else if(GET_STACK_MACHINE_INFO_PROVIDER.equals(methName)) {
			return getStackMachineInformationProvider();
		} else if(CAN_TERMINATE.equals(methName)) {
			return Boolean.valueOf(canTerminate());
		} else if(IS_TERMINATED.equals(methName)) {
			return Boolean.valueOf(isTerminated());
		} else if(TERMINATE.equals(methName)) {
			terminate();
			return null;
		}
		
		throw new InvocationTargetException(new NoSuchMethodException("method \'"+methName + "\' does not exist"));
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.KenyaRemote#canTerminate()
	 */
	public boolean canTerminate() {
		return !hasFinished();
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.KenyaRemote#isTerminated()
	 */
	public boolean isTerminated() {
		return hasFinished();
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.KenyaRemote#terminate()
	 */
	public void terminate() {
		destroy();
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.AbstractVKProcess#getStreamManager()
	 */
	protected IProcessStreamManager getStreamManager() {
		if (fStreamManager == null) {
			fStreamManager = new VKDebugProcessStreamManager();
			
		}

		return fStreamManager;
	}
	
	class VKDebugProcessStreamManager implements IProcessStreamManager {
		
		public void closeInput() throws IOException {
			IOException ee = null;
			try {
				if (inStream != null) {
					inStream.close();
				}
			} catch(IOException e) {
				ee = e;
			}
			if(bufferedIn!=null) {
				bufferedIn.close();
			}
			if(ee!=null) {
				throw ee;
			}
		}
		
		public void closeOutput() throws IOException {
			if(outStream!=null) {
				outStream.close();
			}
			if(bufferedOut!=null) {
				bufferedOut.close();
			}
		}
		
		public void closeErr() throws IOException {
			if(errStream!=null) {
				errStream.close();
			}
			if(bufferedErr!=null) {
				bufferedErr.close();
			}
		}
	}
}
