/*
 * Created on 22-Dec-2004
 */
package kenya.eclipse.debug.model.breakpoints;

import java.util.HashMap;

import kenya.eclipse.debug.model.KenyaDebugModel;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.sourceCodeInformation.util.SourceCodeLocation;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.LineBreakpoint;

/**
 * @author Thomas Timbul
 */
public class KenyaLineBreakpoint extends LineBreakpoint {
	
	protected static final String PROP_PREF = "_KenyaLineBreakPoint.";
	protected static final String REAL_LINE = "real_line";
	protected static final String REAL_COL  = "real_column";
	protected static final String REAL_LEN  = "real_length";
	
	public KenyaLineBreakpoint() {}
	
	public KenyaLineBreakpoint(final IResource resource, final ISourceCodeLocation location, final int charStart, final int charEnd, final boolean register) throws DebugException {
		
		IWorkspaceRunnable wr= new IWorkspaceRunnable() {
			public void run(IProgressMonitor monitor) throws CoreException {
				
				// create the marker
				setMarker(resource.createMarker(getMarkerType()));
				
				HashMap attributes = new HashMap(8);
				
				int lineNumber = location.getLineNumber();
				
				attributes.put( IBreakpoint.ID      , getModelIdentifier() );
				attributes.put( IBreakpoint.ENABLED , Boolean.TRUE         );
				
				attributes.put( IMarker.LINE_NUMBER , new Integer( lineNumber ));
				attributes.put( IMarker.CHAR_START  , new Integer( charStart  ));
				attributes.put( IMarker.CHAR_END    , new Integer( charEnd    ));
				
				attributes.put( REAL_LINE, new Integer( location.getLineNumber()   ));
				attributes.put( REAL_COL , new Integer( location.getColumnNumber() ));
				attributes.put( REAL_LEN , new Integer( location.getTokenLength()  ));
				
				ensureMarker().setAttributes(attributes);
				
				// add to breakpoint manager if requested
				register(register);
			}
		};
		run(null, wr);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IBreakpoint#getModelIdentifier()
	 */
	public String getModelIdentifier() {
		return KenyaDebugModel.getModelIdentifier();
	}
	
	public static String getMarkerType() {
		return "kenya.eclipse.kenyaBreakpointMarker";
	}
	
	/**
	 * returns the real location of this breakpoint as an ISourceCodeLocation.
	 * The values in the location are as returned by the
	 * getRealXYZ methods.
	 * @return real location of this breakpoint
	 * @throws CoreException if marker access fails
	 */
	public ISourceCodeLocation getRealLocation() throws CoreException {
		int line = getRealLineNumber();
		int col = getRealColumnNumber();
		int len = getRealLength();
		return new SourceCodeLocation(line, col, len);
	}
	
	/**
	 * returns the real line number (defined by REAL_LINE) or
	 * getLineNumber() if REAL_LINE attribute is not set.
	 * @return real line number of this breakpoint
	 * @throws CoreException if marker access fails
	 */
	public int getRealLineNumber() throws CoreException {
		return ensureMarker().getAttribute(REAL_LINE, getLineNumber());
	}
	
	/**
	 * returns the real column number (starting character) of this breakpoint
	 * on the line that it was set as defined by REAL_COL. If
	 * the attribute REAL_COL is not set, this method returns 0.
	 * This method is different from getCharStart in that this will
	 * return the offset of the breakpoint on its line, while getCharStart
	 * return the absolute offset in the document.
	 * @return the offset of the breakpoint on its line
	 * @throws CoreException if marker access fails
	 */
	public int getRealColumnNumber() throws CoreException {
		return ensureMarker().getAttribute(REAL_COL, 0);
	}
	
	/**
	 * returns the real length of this breakpoint as defined by REAL_LEN.
	 * If no such attribute is found then the difference between
	 * getCharStart and getcharEnd is returned.
	 * @return the real length of this breakpoint
	 * @throws CoreException if marker access fails
	 */
	public int getRealLength() throws CoreException {
		return ensureMarker().getAttribute(REAL_LEN, getCharStart()-getCharEnd());
	}
	
	/**
	 * Add this breakpoint to the breakpoint manager,
	 * or sets it as unregistered.
	 */
	protected void register(boolean register) throws CoreException {
		if (register) {
			DebugPlugin.getDefault().getBreakpointManager().addBreakpoint(this);
		} else {
			setRegistered(false);
		}
	}	
	
	/**
	 * Execute the given workspace runnable
	 */
	protected void run(ISchedulingRule rule, IWorkspaceRunnable wr) throws DebugException {
		try {
			ResourcesPlugin.getWorkspace().run(wr, rule, 0, null);
		} catch (CoreException e) {
			throw new DebugException(e.getStatus());
		}			
	}
	
}
