/*
 * Created on 17-Dec-2004
 */
package kenya.eclipse.debug.ui.launchconfig;

import java.text.MessageFormat;
import java.util.ArrayList;

import kenya.eclipse.KenyaConstants;
import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.debug.KenyaTypeSearchEngine;
import kenya.eclipse.debug.launcher.IKenyaLaunchConfigurationConstants;
import kenya.eclipse.debug.ui.launcher.TypeSelectionDialog;
import kenya.eclipse.ui.KenyaImages;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.dialogs.SelectionDialog;

/**
 * A launch configuration tab that displays and edits project and
 * main type name launch configuration attributes.
 * <p>
 * This class may be instantiated. This class is not intended to be subclassed.
 * </p>
 * @since 2.0
 */

public class MainTab extends KenyaLaunchConfigurationTab {
	
	// Project UI widgets
	protected Text fProjText;
	protected Button fProjButton;
	
	// Main class UI widgets
	protected Text fMainText;
	protected Button fSearchButton;
	protected Button fStopInMainCheckButton;
	//to hold the actual main path (rather than name) that is stored in config
	protected IPath fMainPath;
	
	/**
	 * A listener which handles widget change events for the controls
	 * in this tab.
	 */
	private class WidgetListener implements ModifyListener, SelectionListener {
		public void modifyText(ModifyEvent e) {
			updateLaunchConfigurationDialog();
		}
		public void widgetSelected(SelectionEvent e) {
			Object source = e.getSource();
			if (source == fProjButton) {
				handleProjectButtonSelected();
			} else if (source == fSearchButton) {
				handleSearchButtonSelected();
			} else {
				updateLaunchConfigurationDialog();
			}
		}
		public void widgetDefaultSelected(SelectionEvent e) {
		}
	}
	
	private WidgetListener fListener = new WidgetListener();
	
	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#createControl(Composite)
	 */
	public void createControl(Composite parent) {
		Font font = parent.getFont();
		
		Composite comp = new Composite(parent, SWT.NONE);
		setControl(comp);
		//WorkbenchHelp.setHelp(getControl(), IJavaDebugHelpContextIds.LAUNCH_CONFIGURATION_DIALOG_MAIN_TAB);
		GridLayout topLayout = new GridLayout();
		topLayout.verticalSpacing = 0;
		topLayout.numColumns = 2;
		comp.setLayout(topLayout);
		comp.setFont(font);
		
		Label projLabel = new Label(comp, SWT.NONE);
		projLabel.setText("Project:");
		GridData gd = new GridData();
		gd.horizontalSpan = 2;
		projLabel.setLayoutData(gd);
		projLabel.setFont(font);
		
		fProjText = new Text(comp, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		fProjText.setLayoutData(gd);
		fProjText.setFont(font);
		fProjText.addModifyListener(fListener);
		
		fProjButton = createPushButton(comp, "Browse...", null);
		fProjButton.addSelectionListener(fListener);
		
		createVerticalSpacer(comp, 2);
		
		createMainTypeEditor(comp);
		
		createVerticalSpacer(comp, 2);
		
		fStopInMainCheckButton = createCheckButton(comp, "Stop in main");
		gd = new GridData();
		gd.horizontalSpan = 2;
		fStopInMainCheckButton.setLayoutData(gd);
		fStopInMainCheckButton.addSelectionListener(fListener);		
		
	}
	
	/**
	 * Creates the widgets for specifying a main type.
	 * 
	 * @param parent the parent composite
	 */
	private void createMainTypeEditor(Composite parent) {
		Font font= parent.getFont();
		Group mainGroup= new Group(parent, SWT.NONE);
		mainGroup.setText("Kenya Type:");
		GridData gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gd.horizontalSpan = 2;
		mainGroup.setLayoutData(gd);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		mainGroup.setLayout(layout);
		mainGroup.setFont(font);
		
		fMainText = new Text(mainGroup, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		fMainText.setLayoutData(gd);
		fMainText.setFont(font);
		fMainText.addModifyListener(fListener);
		
		fSearchButton = createPushButton(mainGroup,"Search...", null);
		fSearchButton.addSelectionListener(fListener);
	}
	
	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	public void initializeFrom(ILaunchConfiguration config) {
		updateProjectFromConfig(config);
		updateMainTypeFromConfig(config);
		updateStopInMainFromConfig(config);
	}
	
	protected void updateProjectFromConfig(ILaunchConfiguration config) {
		String projectName = EMPTY_STRING;
		try {
			projectName = config.getAttribute(IKenyaLaunchConfigurationConstants.ATTR_PROJECT_NAME, EMPTY_STRING);	
		} catch (CoreException ce) {
			KenyaPlugin.log(ce);
		}
		fProjText.setText(projectName);
	}
	
	protected void updateMainTypeFromConfig(ILaunchConfiguration config) {
		String mainTypeName = EMPTY_STRING;
		try {
			mainTypeName = config.getAttribute(IKenyaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME, EMPTY_STRING);
		} catch (CoreException ce) {
			KenyaPlugin.log(ce);
		}
		if(EMPTY_STRING.equals(mainTypeName)) {
			fMainPath = null;
		} else {
			fMainPath = new Path(mainTypeName);
			mainTypeName = fMainPath.removeFileExtension().lastSegment();
			fMainPath = getWorkspaceRoot().getFile(fMainPath).getFullPath();
		}
		fMainText.setText(mainTypeName);	
	}
	
	protected void updateStopInMainFromConfig(ILaunchConfiguration configuration) {
		boolean stop = false;
		try {
			stop = configuration.getAttribute(IKenyaLaunchConfigurationConstants.ATTR_STOP_IN_MAIN, false);
		} catch (CoreException e) {
			KenyaPlugin.log(e);
		}
		fStopInMainCheckButton.setSelection(stop);
	}
	
	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy config) {
		config.setAttribute(IKenyaLaunchConfigurationConstants.ATTR_PROJECT_NAME, fProjText.getText());
		config.setAttribute(IKenyaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME, fMainPath==null?null:fMainPath.toString());
		config.setAttribute(IKenyaLaunchConfigurationConstants.ATTR_STOP_IN_MAIN, fStopInMainCheckButton.getSelection());
	}
	
	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#dispose()
	 */
	public void dispose() {
	}
	
	/**
	 * Show a dialog that lists all main types
	 */
	protected void handleSearchButtonSelected() {
		
		IContainer project = getKenyaProject();
		if(project==null) {
			project = getWorkspaceRoot();
		}
		KenyaTypeSearchEngine engine = new KenyaTypeSearchEngine();
		IPath[] types = null;
		try {
			types = engine.search(project);
		} catch (CoreException e) {
			setErrorMessage(e.getMessage());
			return;
		}
		
		Shell shell = getShell();
		SelectionDialog dialog = new TypeSelectionDialog(shell, types); 
		dialog.setTitle("Choose main type");
		dialog.setMessage("Choose a main type and stuff");
		if (dialog.open() == Window.CANCEL) {
			return;
		}
		
		Object[] results = dialog.getResult();
		if ((results == null) || (results.length < 1)) {
			return;
		}
		fMainPath = ((IPath)results[0]);
		if (fMainPath != null) {
			project = getWorkspaceRoot().getProject(fMainPath.segment(0));
			fProjText.setText(project.getName());
			
			fMainPath = getWorkspaceRoot().getFile(fMainPath).getFullPath();
			fMainText.setText(fMainPath.removeFileExtension().lastSegment());
			
		}
	}
	
	/**
	 * Show a dialog that lets the user select a project.  This in turn provides
	 * context for the main type, allowing the user to key a main type name, or
	 * constraining the search for main types to the specified project.
	 */
	protected void handleProjectButtonSelected() {
		IProject project = chooseProject();
		if (project == null) {
			return;
		}
		
		String projectName = project.getName();
		fProjText.setText(projectName);		
	}
	
	/**
	 * Realize a Kenya Project selection dialog and return the first selected project,
	 * or null if there was none.
	 */
	protected IProject chooseProject() {
		IProject[] projects = new IProject[0];
		try {
			IResource[] r = getWorkspaceRoot().members();
			ArrayList l = new ArrayList(r.length);
			
			for(int i = 0; i < r.length; i++) {
				if(((IProject)r[i]).hasNature(KenyaConstants.K_NATURE_ID)) {
					l.add(r[i]);
				}
			}
			projects = (IProject[])l.toArray(projects);
			
		} catch (CoreException e) {
			KenyaPlugin.log(e);
			//projects = new IProject[0]; //(done in definition)
		}
		
		ILabelProvider labelProvider = new LabelProvider() {
			public String getText(Object element) {
				if(element instanceof IProject) {
					return ((IProject)element).getName();
				}
				return super.getText(element);
				
			}
		};
		
		ElementListSelectionDialog dialog= new ElementListSelectionDialog(getShell(), labelProvider);
		dialog.setTitle("Select a project");
		dialog.setMessage("Select the project that contains the program you wish to launch");
		dialog.setElements(projects);
		
		IProject kenyaProject = getKenyaProject();
		if (kenyaProject != null) {
			dialog.setInitialSelections(new Object[] { kenyaProject });
		}
		if (dialog.open() == Window.OK) {			
			return (IProject)dialog.getFirstResult();
		}			
		return null;		
	}
	
	/**
	 * Return the IProject corresponding to the project name in the project name
	 * text field or null if the text does not match a project name.
	 */
	protected IProject getKenyaProject() {
		String projectName = fProjText.getText().trim();
		if (projectName.length() < 1) {
			return null;
		}
		return getWorkspaceRoot().getProject(projectName);
	}
	
	/**
	 * Convenience method to get the workspace root.
	 */
	private IWorkspaceRoot getWorkspaceRoot() {
		return ResourcesPlugin.getWorkspace().getRoot();
	}
	
	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#isValid(ILaunchConfiguration)
	 */
	public boolean isValid(ILaunchConfiguration config) {
		
		setErrorMessage(null);
		setMessage(null);
		
		String name = fProjText.getText().trim();
		if (name.length() > 0) {
			IWorkspace workspace = ResourcesPlugin.getWorkspace();
			IStatus status = workspace.validateName(name, IResource.PROJECT);
			if (status.isOK()) {
				IProject project= ResourcesPlugin.getWorkspace().getRoot().getProject(name);
				if (!project.exists()) {
					setErrorMessage(MessageFormat.format("Project {0} does not exist", new String[] {name}));
					return false;
				}
				if (!project.isOpen()) {
					setErrorMessage(MessageFormat.format("Project {0} is closed", new String[] {name}));
					return false;
				}
			} else {
				setErrorMessage(MessageFormat.format("Illegal project name: {0}", new String[]{status.getMessage()}));
				return false;
			}
		}
		
		name = fMainText.getText().trim();
		if (name.length() == 0) {
			setErrorMessage("Main type not specified");
			return false;
		}
		
		return true;
	}
	
	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#setDefaults(ILaunchConfigurationWorkingCopy)
	 */
	public void setDefaults(ILaunchConfigurationWorkingCopy config) {
		IResource javaElement = getContext();
		if (javaElement != null) {
			initializeKenyaProject(javaElement, config);
		} else {
			config.setAttribute(IKenyaLaunchConfigurationConstants.ATTR_PROJECT_NAME, EMPTY_STRING);
		}
		initializeMainTypeAndName(javaElement, config);
	}
	
	/**
	 * Set the main type & name attributes on the working copy based on the main type resource
	 */
	protected void initializeMainTypeAndName(IResource mainType, ILaunchConfigurationWorkingCopy config) {
		IPath path = mainType!=null?mainType.getFullPath():null;
		String name = path!=null?path.removeFileExtension().lastSegment():EMPTY_STRING;
		
		String pathString = path!=null?path.toString():EMPTY_STRING;
		config.setAttribute(IKenyaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME, pathString);
		if (name.length() > 0) {
			name = getLaunchConfigurationDialog().generateName(name);
			config.rename(name);
		}
	}
	
	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#getName()
	 */
	public String getName() {
		return "Main";
	}
	
	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#getImage()
	 */
	public Image getImage() {
		return KenyaImages.getImage(KenyaImages.IMG_OBJ_CLASS);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#activated(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void activated(ILaunchConfigurationWorkingCopy workingCopy) {
		// do nothing when activated
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#deactivated(org.eclipse.debug.core.ILaunchConfigurationWorkingCopy)
	 */
	public void deactivated(ILaunchConfigurationWorkingCopy workingCopy) {
		// do nothing when deactivated
	}
}

