package kenya.eclipse.multieditor.kenya.refactoring;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import kenya.eclipse.KenyaPlugin;

import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;


/**
 * Operation that can be applied to an IDocument. The use of this is to
 * add multiple independent DocumentTextOperations that will all be applied
 * in a single step without having to worry about their offsets (which
 * would change after applying each one).
 * This implementation does not take care of overlapping DocumentTextOperations.
 * 
 * @author Thomas Timbul
 */
public class DocumentModificationOperation {
	
	protected SortedOperationList fOperations;
	protected Map fRequiredInput; // <String,String>
	protected Map fRequiredValidation; // <String,IInputValidator>
	
	/**
	 * the input Map provided maps input prompts such as 'Enter xyz...' to
	 * a variable name. When run() is invoked a dialog with that title
	 * is presented to the user and the variables mapped. The result is
	 * then supplied as an argument to each of the
	 * DocumentTextOperation.perform(IDocument, Map) methods.
	 * inputsRequired can be null if no inputs are required.
	 * The validation Map maps from the variable name given in the input Map
	 * to an instance of IInputValidator that is used to validate user input.
	 * The mapping is not required if no input validation is required.
	 * Therefore if validation is not required for any of the inputs, then
	 * this map will be ignored and can be null.
	 * 
	 * @param inputsRequired
	 * @param validationRequired
	 */
	public DocumentModificationOperation(Map inputsRequired, Map validationRequired) {
		fOperations = new SortedOperationList();
		setRequiredInput(inputsRequired);
		setRequiredValidation(validationRequired);
	}
	
	/**
	 * same as calling <code>DocumentModificationOperation(null, null)</code>
	 */
	public DocumentModificationOperation() {
		this(null, null);
	}
	
	public void setRequiredInput(Map inputsRequired) {
		fRequiredInput = (inputsRequired!=null)
		      ?inputsRequired
		      :new HashMap(0);
	}
	
	public void setRequiredValidation(Map validationRequired) {
		fRequiredValidation = (validationRequired!=null)
		      ?validationRequired
		      :new HashMap(0);
	}
	
	public void addOperation(DocumentTextOperation op) {
		fOperations.add(op);
	}
	
	/**
	 * first maps each of the variables given as values in the Map that
	 * was supplied to the constructor, then iteratively calls perform(IDocument, Map)
	 * with the given IDocument and the resulting Map of variables on each
	 * DocumentTextOperation that was previously added using addOperation(DocumentTextOperation).
	 * @param d the document to run this operation on
	 */
	public void run(IDocument doc) {
		HashMap replacements = new HashMap(fRequiredInput.size());
		for(Iterator it = fRequiredInput.entrySet().iterator(); it.hasNext();) {
			Map.Entry entry = (Map.Entry)it.next();
			
			IInputValidator validator
			  = (IInputValidator)fRequiredValidation.get(entry.getValue());
			
			InputDialog d
			  = new InputDialog(KenyaPlugin.getActiveWorkbenchShell(),
			  		"Input required..", entry.getKey().toString(),
						"",
						validator
					);
			d.setBlockOnOpen(true);
			
			d.create();
			int choice = d.open();
			if(choice != InputDialog.OK) {
				MessageDialog.openInformation(KenyaPlugin.getActiveWorkbenchShell(),
						"Cancelled",
						"You have chosen to cancel the process,\nno changes will be performed");
				return;
			}
			
			String newId = d.getValue();
			
			replacements.put(entry.getValue(), newId);
			
		}
		
		DocumentTextOperation[] ops = fOperations.toArray();
		for(int i = ops.length-1; i >= 0; i--) {
			//back to front, do later ones first
			// that way we avoid shifting of offsets!
			DocumentTextOperation op = ops[i];
			try {
				op.perform(doc, replacements);
			} catch(BadLocationException e) {
			}
		}
	}
	
}
