/*
 * Created on 10-Feb-2005
 */
package kenya.eclipse.multieditor.kenya.refactoring;

import java.util.Iterator;
import java.util.Map;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;

/**
 * @author Thomas Timbul
 */
public class DocumentTextOperation {
	
	protected static final String EMPTY = "";
	
	protected String fReplaceWith;
	protected int fOffset;
	protected int fLength;
	
	protected DocumentTextOperation(int offset, int length, String replaceWith) {
		fReplaceWith = replaceWith;
		fOffset = offset;
		fLength = length;
	}
	
	/**
	 * performs this operation on the given document with the given Map of
	 * variable substitutions. Replacable variables can be defined in
	 * the inserted or replacement String of the according DocumentTextOperation by
	 * enclosing the variable with Ant style notation such as
	 * <code>${var_name}</code>.
	 * The Map maps var_name (<b>without</b> the additional notation)
	 * to the String it should be replaced by. If an Object other than String
	 * is encountered in the Map, its toString value shall be used instead.
	 * 
	 * The replacement is done dynamically for each call of this method.
	 * It is therefore technically possible to use the same operation multiple times, but
	 * with different substitutions, although a practical application of this
	 * is possibly doubtful. Consideration is to be taken if this method is to be
	 * invoked a lot of times, with many variables to substitute,
	 * in which case it would be better to not use variable substitution, but
	 * instead construct the insert or replace in its final form and supply an
	 * empty or null Map.
	 * 
	 * @param doc the IDocument to perform this operation on
	 * @param varReplacements a Map defining variable substitutions
	 * @throws BadLocationException if the operation refers to an invalid
	 * location in the document
	 */
	public void perform(IDocument doc, Map varReplacements) throws BadLocationException {
		String actualReplace = fReplaceWith;
		if(varReplacements!=null) {
			for(Iterator it = varReplacements.entrySet().iterator(); it.hasNext();) {
				Map.Entry entry = (Map.Entry)it.next();
				String varPattern = "\\$\\{\\Q" + entry.getKey().toString() + "\\E\\}";
				String sub = entry.getValue().toString();
				actualReplace = actualReplace.replaceAll(varPattern, sub);
			}
		}
		
		doc.replace(fOffset, fLength, actualReplace);
	}
	
	public static DocumentTextOperation newTextDeletion(int offset, int length) {
		return new DocumentTextOperation(offset, length, EMPTY);
	}
	
	public static DocumentTextOperation newTextInsertion(int offset, String insert) {
		return new DocumentTextOperation(offset, 0, insert);
	}
	
	public static DocumentTextOperation newTextReplacement(int offset, int length, String replacement) {
		return new DocumentTextOperation(offset, length, replacement);
	}
}
