/*
 * Created on 17-Feb-2005
 */
package kenya.eclipse.multieditor.kenya.util;

import kenya.eclipse.ast.NodeTools;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.sourceCodeInformation.util.SourceCodeLocation;
import minijava.node.Node;
import minijava.node.Token;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.Position;


/**
 * Utilities that relate to ISourceCodeLocations, Positions and IDocuments
 * 
 * @author Thomas Timbul
 */
public class LocationUtils {

	private LocationUtils() {
		//this class needs not be instantiated
	}
	
	/**
	 * returns the location of the given Token as an ISourceCodeLocation
	 * @param t the Token whose location to obtain
	 * @return the ISourceCodeLocation of the given Token
	 */
	public static ISourceCodeLocation getLocation(Token t) {
		return new SourceCodeLocation(t.getLine(), t.getPos(), t.getText().length());
	}
	
	/**
	 * converts the given Position into an ISourceCodeLocation based
	 * on the given IDocument taking care of index conversion (+1).
	 * Result is null if the Position is invalid in the Document
	 * @param pos
	 * @param doc
	 * @return an ISourceCodeLocation equivalent to the given Position or null
	 *         if the Position is invalid
	 */
	public static ISourceCodeLocation convert(Position pos, IDocument doc) {
		try {
			int line = doc.getLineOfOffset(pos.getOffset());
			int col = pos.getOffset()-doc.getLineOffset(line);
			int len = pos.getLength();
			//add one for index correction
			return new SourceCodeLocation(line+1, col+1, len);
		} catch(BadLocationException e) {
			return null;
		}
	}
	
	/**
	 * Returns the offset and length in the given IDocument as an ISourceCodeLocation.
	 * The call is equivalent to
	 * <code>convert(new Position(offset, length), doc);</code>
	 * @see #convert(Position, IDocument)
	 */
	public static ISourceCodeLocation convert(int offset, int length, IDocument doc) {
		return convert(new Position(offset, length), doc);
	}
	
	/**
	 * Calculates and returns the position of an ISourceCodeInformation
	 * in the given document. This method first calculates the offset
	 * of the line of the Information. This takes into account the fact
	 * that Information lines are indexed from 1, whereas the offset is
	 * calculated indexed from 0. The same is true for the column number
	 * which is used to calculate the offset of the Information from
	 * the beginning of the line.
	 * 
	 * @param info the ISourceCodeInformation whose position is to be calculated
	 * @param doc the IDocument in which to calculate the Position
	 * @return the Position of the ISourceCodeInformation in the document
	 *         or null if the location is invalid in the document
	 */
	public static Position convert(ISourceCodeLocation loc, IDocument doc) {
		try {
				//subtract 1 off line number for indexing
			IRegion reg = doc.getLineInformation(loc.getLineNumber()-1);
			return new Position(
					//subtract 1 off col number for indexing
					reg.getOffset() + Math.max(loc.getColumnNumber()-1, 0),
					loc.getTokenLength()
				);
		} catch(BadLocationException e) {
			return null;
		}
		
	}
	
	
	public static boolean occursBefore(Node n1, Node n2) {
		ISourceCodeLocation l1 = NodeTools.getLocation(n1);
		ISourceCodeLocation l2 = NodeTools.getLocation(n2);
		return occursBefore(l1, l2);
	}
	
	public static boolean occursBefore(ISourceCodeLocation l1, ISourceCodeLocation l2) {
		return SourceCodeLocationComparator.getInstance().compare(l1, l2)<=0;
	}
	
	
}
