/*
 * Created on 08-Jan-2005
 */
package kenya.eclipse.style.checkerimpl;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.ast.NodeTools;
import kenya.eclipse.buildext.util.MarkerPropertySetter;
import kenya.eclipse.multieditor.kenya.util.LocationUtils;
import kenya.eclipse.style.KenyaStyleConstants;
import kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.ICheckedCode;
import minijava.node.Node;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.texteditor.DocumentProviderRegistry;
import org.eclipse.ui.texteditor.IDocumentProvider;

/**
 * @author Thomas Timbul
 */
public abstract class AbstractStyleChecker implements IStyleChecker {
	
	public static final String KENYA_STYLE_MARKER = KenyaStyleConstants.KENYA_STYLE_MARKER;
	
	private IStyleCheckerDescriptor fDescriptor;
	
	/**
	 * map of created markers to corresponding MarkerResolution[].
	 * should be populated by subclasses to make quick fix possible in the platform
	 */
	protected Map fResolutionMap;
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.IStyleChecker#setDescriptor(kenya.eclipse.style.IStyleCheckerDescriptor)
	 */
	public void setDescriptor(IStyleCheckerDescriptor descriptor) {
		fDescriptor = descriptor;
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.IStyleChecker#getDescriptor()
	 */
	public IStyleCheckerDescriptor getDescriptor() {
		return fDescriptor;
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#getMarkerResolutionMap()
	 */
	public Map getMarkerResolutionMap() {
		return fResolutionMap!=null?fResolutionMap:new HashMap(0);
	}
	
	/**
	 * Checks whether the given node overlaps with an error in the given
	 * source code object.<br>
	 * The result is true if there is an overlap and false if there is not.
	 * This method immediately returns true if there are no errors in the code,
	 * otherwise it will iterate through all the errors and check each one for
	 * and overlap with the given node.
	 * <br>An overlap occurs if one of the following is true:
	 * <ul>
	 *     <li>error begins inside the node's location 
	 *     <li>error ends inside the node's location
	 *     <li>node begins inside the error's location
	 *     <li>node ends inside the error's location
	 * </ul>
	 * @param node the node to check for overlap
	 * @param code the code whose errors to check for overlap
	 * @return true if there is an overlap of the node with an error, false otherwise
	 */
	public static boolean overlapsWithError(Node node, ICheckedCode code) {
		if(!code.isErroredCode()) {
			return false;
		} else {
			ISourceCodeLocation loc = NodeTools.getLocation(node);
			int line = loc.getLineNumber();
			int col = loc.getColumnNumber();
			int end = col + loc.getTokenLength();
			
			List errs = code.getErrors();
			for(Iterator iter = errs.iterator(); iter.hasNext();) {
				ISourceCodeError element = (ISourceCodeError)iter.next();
				ISourceCodeLocation errLoc = element.getLocation();
				
				int eLine = errLoc.getLineNumber();
				int eCol = errLoc.getColumnNumber();
				int eEnd = eCol + errLoc.getTokenLength();
				
				if(line==eLine) {
					
					if(
							(col <= eCol && eCol <= end) ||
							(eCol <= col && col <= eEnd) ||
							(col <= eEnd && eEnd <= end) ||
							(eCol <= end && end <= eEnd)
							) {
						return true;
					}
					
				}
				
			}
			return false;
		}
		
	}
	
	/**
	 * Execute the given workspace runnable for a marker update.
	 * There are no SchedulingRules and no ProgressMonitor. The AVOID_UPDATE flag is set.
	 * This method should be used for creating markers from within StyleCheckers.
	 */
	protected static void runMarkerUpdate(IWorkspaceRunnable wr) {
		try {
			ResourcesPlugin.getWorkspace().run(wr, null, IWorkspace.AVOID_UPDATE, null);
		} catch(CoreException e) {
			KenyaPlugin.log(e);
		}
	}
	
	/**
	 * finds the IDocument belonging to this file
	 * @param file
	 * @return
	 */
	protected static IDocument getDocument(IFile file) {
		IFileEditorInput fei = new FileEditorInput(file);
		IDocumentProvider provider
		  = DocumentProviderRegistry.getDefault().getDocumentProvider(fei);
		IDocument doc = provider.getDocument(fei);
		return doc;
	}
	
	/**
	 * calculates the Position (jface) of the given ISourceCodeLocation in the given document
	 * @param loc
	 * @param doc
	 * @return
	 */
	protected static Position calculatePosition(ISourceCodeLocation loc, IDocument doc) {
		return LocationUtils.convert(loc, doc);
	}
	
	/**
	 * creates a style marker at the given location in the given file with the given message.
	 * @param file file on which to create the marker
	 * @param loc the location in the code where to set the marker
	 * @param message the message accompanying this marker
	 * @return the new marker
	 * @throws CoreException if marker creation or setting its attributes fail
	 */
	protected static IMarker createKenyaStyleMarker(IFile file, ISourceCodeLocation loc, String message) throws CoreException {
		
		IMarker m = file.createMarker(KENYA_STYLE_MARKER);
		
		//determine location and position
		IDocument doc = getDocument(file);
		Position pos = calculatePosition(loc, doc);
		
		//create attributes
		HashMap atts = new HashMap(4);
		atts.put(IMarker.SEVERITY, new Integer(IMarker.SEVERITY_WARNING));
		atts.put(IMarker.LINE_NUMBER, new Integer(loc.getLineNumber()));
		atts.put(IMarker.USER_EDITABLE, Boolean.FALSE);
		
		atts.put(IMarker.MESSAGE, message);
		
		m.setAttributes(atts);
		
		MarkerPropertySetter.setLocation(m, pos);
		
		return m;
	}
	
}
