/*
 * Created on 15-Mar-2005
 */
package kenya.eclipse.style.checks.arrays;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import kenya.eclipse.ast.AdvancedPositionFinder;
import kenya.eclipse.ast.NodeToString;
import kenya.eclipse.ast.NodeTools;
import kenya.eclipse.ast.OccurrencesFinder;
import kenya.eclipse.ast.SimpleASTProvider;
import kenya.eclipse.multieditor.kenya.util.SourceCodeLocationComparator;
import kenya.eclipse.style.checkerimpl.AbstractStyleChecker;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.sourceCodeInformation.util.SourceCodeLocation;
import mediator.ICheckedCode;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.AArrayDecInnerDeclaration;
import minijava.node.AArrayFieldAccess;
import minijava.node.AArrayInit;
import minijava.node.AArrayInitList;
import minijava.node.AAssignment;
import minijava.node.ACommaArrayInit;
import minijava.node.ADynamicArrayInitialiser;
import minijava.node.ANormalArrayInitialiser;
import minijava.node.AScalarFieldAccess;
import minijava.node.AScalarInitList;
import minijava.node.AStaticArrayInitialiser;
import minijava.node.Node;
import minijava.node.PArrayInitialiser;
import minijava.node.PAssignment;
import minijava.node.PFieldAccess;
import minijava.node.PInitList;
import minijava.node.Start;
import minijava.node.TIdentifier;

import org.eclipse.core.resources.IFile;
import org.eclipse.jface.text.IDocument;

/**
 * Checks for ArrayIndexOutOfBounds
 * Targeted at direct array access using integers.
 * Access through non-integers (variables) is ignored.
 * 
 * @author Thomas Timbul
 */
public class IndexOutOfBoundsChecker extends AbstractStyleChecker {
	
	private static final Pattern NAME = Pattern.compile("\\s*[\\w\\.\\d]+"); //qualified or simple or just numbers
	private static final Pattern ARRAY_ACCESS = Pattern.compile("\\[\\s*(\\d+)?("+NAME.pattern()+")?\\s*\\]"); //2 groups
	private static final Pattern SCALAR_DECLARATION = Pattern.compile("\\s*(\\{.*\\})\\s*", Pattern.DOTALL); //3 groups
	private static final Pattern DYN_DECLARATION = Pattern.compile("new\\s+[\\w\\d]+\\s*("+ARRAY_ACCESS.pattern()+"*)+"); //3 groups
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#configure(java.util.Map)
	 */
	public void configure(Map customAttributes) {
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#performCheck(mediator.ICheckedCode, org.eclipse.core.resources.IFile)
	 */
	public void performCheck(ICheckedCode code, IFile file) {
		
		IDocument doc = getDocument(file);
		Start ast = SimpleASTProvider.getAST(code);
		
		ArrayList arrayDeclarations = findAllArrayDeclarations(code); // <AArrayDecInnerDeclaration>
		
		for(Iterator it = arrayDeclarations.iterator(); it.hasNext();) {
			AArrayDecInnerDeclaration dec = (AArrayDecInnerDeclaration)it.next();
			System.out.println("\n----------");
			System.out.println("array declaration: " + dec);
			OccurrencesFinder f = new OccurrencesFinder();
			String m = f.initialize(ast, code, doc, AdvancedPositionFinder.getFullLocation(dec.getIdentifier(), doc));
			if(m!=null) continue;
			List ids = f.perform(); // <TIdentifier>
			
			ArrayList locs = new ArrayList(); //will remain empty if array not initialised
			HashMap locToLen = new HashMap(); // <ISourceCodeLocation,String[]>
			
			/* cannot tell if:
			 * allocated inside if/switch/for/while
			 * RHS of assignment is another variable or method call
			 * RHS of assigment is null
			 */
			
			ArrayList usages = new ArrayList();
			
			for(Iterator iter = ids.iterator(); iter.hasNext();) {
				TIdentifier id = (TIdentifier)iter.next();
				
				// check the usage of this identifier (r access or w access/initialisation)
				
				System.out.println(id.parent().parent().parent());
				
				AAssignment assign
				  = (AAssignment)NodeTools.getParent(id, PAssignment.class);
				//if(assign!=null) System.out.print(" - Assignment");
				
				PFieldAccess fieldaccess
				  = (PFieldAccess)NodeTools.getParent(id, PFieldAccess.class);
				
				if(fieldaccess instanceof AArrayFieldAccess) {
					//these are potentially broken, so save for later inspection
					usages.add(fieldaccess);
					
				} else if(assign!=null) {
					if(assign.getFieldAccess()==fieldaccess) {
						// array is LHS of the assignment
						
						if(fieldaccess instanceof AScalarFieldAccess) {
							//an assignment to the array itself - worth checking out
//						but only if its not otherwise assigned, as scalar would be the 'original' declaration - least precedence
							ISourceCodeLocation loc = AdvancedPositionFinder.getFullLocation(assign, doc);
							locs.add(loc);
							locToLen.put(loc, getDeclaredLength(assign.getExpression()));
						}
						
					} else {
						//already covered
					}
					
				} else {
					AArrayDecInnerDeclaration decl
					  = (AArrayDecInnerDeclaration)NodeTools.getParent(id, AArrayDecInnerDeclaration.class);
					if(decl!=null) {
						PArrayInitialiser initialiser = decl.getArrayInitialiser();
							if (initialiser!=null) {
								ISourceCodeLocation loc = new SourceCodeLocation(1, 1, 0);
								locs.add(loc);
								locToLen.put(loc, getDeclaredLength(initialiser));
							}
					} else {
						System.out.println("uncovered case");
					}
					
				}
			}
			
			ISourceCodeLocation[] locations
			  = (ISourceCodeLocation[])locs.toArray(new ISourceCodeLocation[locs.size()]);
			Arrays.sort(locations, SourceCodeLocationComparator.getInstance());
			
			if(locations.length==0) {
				continue; //not declared anywhere!
			}
			System.out.println("accesses:");
			
			for(Iterator it1 = usages.iterator(); it1.hasNext();) {
				AArrayFieldAccess acc = (AArrayFieldAccess)it1.next();
				
				ISourceCodeLocation currentLoc
				  = AdvancedPositionFinder.getFullLocation(acc, doc);
				
				ISourceCodeLocation loc = getJustBefore(locations, currentLoc);
				String[] length = (String[])locToLen.get(loc); //THE length reference
				
				String[] g = getAccessLength(acc);
				for(int i = 0; i < g.length; i++) {
					System.out.print(g[i]+"/");
					
					if(i<length.length) System.out.print(length[i]+"  ");
				}
				
			}
		}
		
	}
	
	//can return null if locs is empty
	private ISourceCodeLocation getJustBefore(ISourceCodeLocation[] locs, ISourceCodeLocation l) {
		ISourceCodeLocation closest = null;
		SourceCodeLocationComparator c = SourceCodeLocationComparator.getInstance();
		for(int i = 0; i < locs.length; i++) {
			if(c.compare(locs[i], l)>0) break;
			closest = locs[i];
		}
		return closest;
	}
	
	//returns array of access values for each dimension
	private String[] getAccessLength(Node n) {
		String txt = NodeToString.toString(n);
		Pattern p = Pattern.compile(NAME.pattern()+"("+ARRAY_ACCESS.pattern()+")+");
		
		//cut off the identifier, so only array access is left
		Matcher m = p.matcher(txt);
		if(m.matches()) {
			txt = txt.substring(m.start(1));
			return getArrayAccessLength(txt);
		}
		
		//if that is not possible, then no idea - empty array
		return new String[0];
	}
	
	//returns array of access values for each dimension
	private String[] getDeclaredLength(Node n) {
		
		if(n instanceof AStaticArrayInitialiser) {
			AStaticArrayInitialiser in = (AStaticArrayInitialiser)n;
			AArrayInit init = (AArrayInit)in.getArrayInit();
			int d = getNumberOfDimensions(init);
			PInitList list = init.getInitList();
			int s = getLength((AArrayInitList)init.getInitList());
		}
		
		String txt = NodeToString.toString(n);
		return getDeclaredLength(txt);
	}
	
	private int getNumberOfDimensions(AArrayInit init) {
		PInitList list = init.getInitList();
		if(list instanceof AArrayInitList) {
			int s = 1;
			AArrayInitList l = (AArrayInitList)list;
			AArrayInit in = (AArrayInit)l.getArrayInit();
			s += getNumberOfDimensions(in);
			LinkedList abc = l.getCommaArrayInit();
			if(l!=null) {
				for(Iterator it = abc.iterator(); it.hasNext();) {
					ACommaArrayInit element = (ACommaArrayInit)it.next();
					in = (AArrayInit)element.getArrayInit();
					s += getNumberOfDimensions(in);
				}
			}
			return s;
		} else if(list instanceof AScalarInitList) {
			return 1;
		}
		
			
		return 0;
	}
	
	private int getLength(AArrayInitList list) {
		return list.getCommaArrayInit()!=null
		  ?list.getCommaArrayInit().size()+1:1;
	}
	
	//returns each dimension's declared length
	private String[] getDeclaredLength(String txt) {
		Matcher m = DYN_DECLARATION.matcher(txt);
		if(m.matches()) {
			return getArrayAccessLength(m.group(1));
		}
		
		return new String[0];
	}
	
	private String[] getArrayAccessLength(String txt) {
		Matcher m = ARRAY_ACCESS.matcher(txt);
		
		ArrayList list = new ArrayList();
		while(m.find()) {
			if(m.group(1)!=null) {
				list.add(m.group(1));
			} else {
				list.add(m.group(2)); //can potentially be null
			}
		}
		return (String[])list.toArray(new String[list.size()]);
	}
	
	private ArrayList findAllArrayDeclarations(ICheckedCode code) {
		return new ArrayDeclarationFinder().perform(SimpleASTProvider.getAST(code));
	}
	
}


class ArrayDeclarationFinder extends DepthFirstAdapter {
	
	private ArrayList list;
	
	/**
	 * @param ast
	 * @return
	 */
	public ArrayList perform(Start ast) {
		list = new ArrayList();
		ast.apply(this);
		return list;
	}
	
	/* (non-Javadoc)
	 * @see minijava.analysis.DepthFirstAdapter#caseAArrayDecInnerDeclaration(minijava.node.AArrayDecInnerDeclaration)
	 */
	public void caseAArrayDecInnerDeclaration(AArrayDecInnerDeclaration node) {
		list.add(node);
	}
	
}


class ArrayAccessFinder extends DepthFirstAdapter {
	
	private ArrayList list;
	
	/**
	 * @param ast
	 * @return
	 */
	public ArrayList perform(Start ast) {
		list = new ArrayList();
		ast.apply(this);
		return list;
	}
	
	/* (non-Javadoc)
	 * @see minijava.analysis.DepthFirstAdapter#caseAArrayFieldAccess(minijava.node.AArrayFieldAccess)
	 */
	public void caseAArrayFieldAccess(AArrayFieldAccess node) {
		list.add(node);
	}
	
	
}


class ArrayInitFinder extends DepthFirstAdapter {
	
	private ArrayList list;
	
	/**
	 * @param ast
	 * @return
	 */
	public ArrayList perform(Start ast) {
		list = new ArrayList();
		ast.apply(this);
		return list;
	}
	
	public void caseADynamicArrayInitialiser(ADynamicArrayInitialiser node) {
		list.add(node);
	}
	
	public void caseAStaticArrayInitialiser(AStaticArrayInitialiser node) {
		list.add(node);
	}
	
	public void caseANormalArrayInitialiser(ANormalArrayInitialiser node) {
		list.add(node);
	}
	
}






