/*
 * Created on 20-May-2005
 */
package kenya.eclipse.style.checks.metrics;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import kenya.eclipse.ast.AdvancedPositionFinder;
import kenya.eclipse.style.AutomatedNAVStyleWarningResolution;
import kenya.eclipse.style.StyleWarningResolution;
import kenya.eclipse.style.checkerimpl.AbstractStyleChecker;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.ICheckedCode;
import minijava.node.AFormalParamList;
import minijava.node.AFuncDecDeclaration;
import minijava.node.Node;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.text.IDocument;

/**
 * @author Thomas Timbul
 */
public class ParameterLengthChecker extends AbstractStyleChecker {
	
	public static final String MESSAGE =
		"This method takes {0} parameters. " +
		"\nTry splitting the method into smaller functional parts or creating" +
		"\na class to group closely related parameters.";
	
	//configuration key - the maximum number of parameters to allow
	private static final String lengthKey = "length";
	private int fMaxLength = Integer.MAX_VALUE; //initial value is 'infinite'
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#configure(java.util.Map)
	 */
	public void configure(Map customAttributes) {
		String ln = (String)customAttributes.get(lengthKey);
		if(ln!=null) {
			try {
				int length = Integer.parseInt(ln);
				fMaxLength = length;
				//retrieve the length from the configuration
			} catch(NumberFormatException e) {
				//ignore (if fails, we will be using the default value)
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#performCheck(mediator.ICheckedCode, org.eclipse.core.resources.IFile)
	 */
	public void performCheck(ICheckedCode code, final IFile file) {
		
		IDocument doc = getDocument(file);
		
		//step 1
		IFunction[] funs = code.getFunctions();
		
		//lengths will contain the offending method's location
		// and the number of arguments they accept
		HashMap lengths = new HashMap();
		
		//we first examine each function in turn
		for(int i = 0; i < funs.length; i++) {
			IFunction function = funs[i];
			AFuncDecDeclaration decl = function.getDeclarationNode();
			
			//the following is cheating as it requires implicit knowledge of the AST
			// the proper way is to use a visitor that counts the number of parameters
			AFormalParamList paramlist = (AFormalParamList)decl.getFormalParamList();
			if(paramlist==null) {
				continue; //no parameters
			}
			Node n = paramlist.getTypeName();
			List list = paramlist.getCommaTypeName();
			
			int length = (n==null)
			  ?0                 //if n is null, length is 0
			  :1 + (             //otherwise 1 for n, and add
			    (list==null)
			        ?0           //0 for a null list
			        :list.size() //or its length
			     );
			
			if(length>fMaxLength) {
				//we actually wish to highlight the parameter list
				ISourceCodeLocation loc
				  = AdvancedPositionFinder.getFullLocation(paramlist, doc);
				lengths.put(loc, new Integer(length));
			}
			
		} //this concludes step 2
		
		//we can now go through the 'offenders' and create the appropriate markers
		//this must be done in an IWorkSpaceRunnable()
		final Set offences = lengths.entrySet();
		
		IWorkspaceRunnable runnable = new IWorkspaceRunnable() {
			public void run(IProgressMonitor monitor) throws CoreException {
				for(Iterator it = offences.iterator(); it.hasNext();) {
					Map.Entry entry = (Map.Entry)it.next();
					
					//retrieve the data from the map
					ISourceCodeLocation loc = (ISourceCodeLocation)entry.getKey();
					Integer length = (Integer)entry.getValue();
					
					String msg = MessageFormat.format(MESSAGE, new Object[]{length});
					
					//super can do the marker creation
					IMarker m = createKenyaStyleMarker(file, loc, msg);
					createResolution(m);
					
				}
			}
		};
		
		//execute the runnable
		runMarkerUpdate(runnable);
		
	}
	
	protected void createResolution(IMarker marker) {
		StyleWarningResolution res = new AutomatedNAVStyleWarningResolution(marker) {
			public String getDescription() {
				return "<p><b>Explanation:</b></p>" +
						"<p>The method in question takes many parameters. " +
						"Sometimes this is unavoidable, however you should try to keep methods as " +
						"short as possible and let them do as little as possible. This ensures " +
						"that you have many small reusable components rather than repeated blocks " +
						"of code, which also makes debugging much easier.</p>" +
						"<p>If you find that you are often using the same group of variables together, " +
						"then you should " +
						"consider creating a class which contains logically related variables.</p>";
			}
		};
		if(fResolutionMap==null) {
			fResolutionMap = new HashMap();
		}
		fResolutionMap.put(marker, new StyleWarningResolution[]{res});
	}
	
}
