/*
 * Created on 08-Jan-2005
 */
package kenya.eclipse.style.properties;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import kenya.eclipse.style.KenyaStyleConstants;
import kenya.eclipse.style.KenyaStyleManager;
import kenya.eclipse.style.StyleCheckRegistry;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.QualifiedName;

/**
 * @author Thomas Timbul
 */
public class PropertyUtils {
	
	public static final String ENABLED = KenyaStyleConstants.ENABLED;
	public static final String DISABLED = KenyaStyleConstants.DISABLED;
	
	/**
	 * returns a filtered list of checks on the specified file.
	 * filter should be one of ENABLED or DISABLED, an empty list is returned if
	 * this is not the case.
	 * setDefaults specifies whether new styles (where properties have not been set)
	 * should be initialised to the default value (enabled)
	 * 
	 * @param file the IFile for which to gather check information
	 * @param filter whether enabled or disabled checks should be returned
	 * @param setDefaults whether or not checks whose enabled property has not
	 *  been set on the file should be set to enabled
	 * @return a List (ArrayList) containing all IStyleCheckers that match the given criteria
	 */
	public static List getFilteredEnabledStyleChecks(IFile file, String filter, boolean setDefaults) {
		
		StyleCheckRegistry reg = KenyaStyleManager.getStyleCheckRegistry();
		
		Collection ids = reg.listIds();
		ArrayList al = new ArrayList();
		
		if(filter!=ENABLED && filter!=DISABLED) {
			return al;
		}
		
		try {
			for(Iterator it = ids.iterator(); it.hasNext();) {
				QualifiedName id = (QualifiedName)it.next();
				
				String val = file.getPersistentProperty(id);
				
				if(val==null && setDefaults) {
					val = reg.getStyleChecker(id).getDescriptor().isDefaultEnabled()
						?ENABLED
						:DISABLED;
					file.setPersistentProperty(id, val);
				}
				
				if(filter.equals(val)) {
					al.add(reg.getStyleChecker(id));
				}
				
			}
		} catch(CoreException e) {
			//ignore, we will have to abort in this case...
		}
		
		return al;
	}
	
	public static boolean isStyleCheckingEnabled(IFile file, boolean setDefaults) {
		try {
			String s = file.getPersistentProperty(KenyaStyleConstants.PROP_STYLE_CHECKING_ENABLED);
			if(s==null && setDefaults) {
				s = ENABLED;
				file.setPersistentProperty(KenyaStyleConstants.PROP_STYLE_CHECKING_ENABLED, s);
			}
			
			return ENABLED.equals(s);
			
		} catch(CoreException e) {
			//ignore
			return false;
		}
		
	}
	
}
