/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package kenya.eclipse.ui.actions;

import java.util.Iterator;
import java.util.List;

import kenya.eclipse.KenyaConventions;
import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.ast.AdvancedPositionFinder;
import kenya.eclipse.ast.IBinding;
import kenya.eclipse.ast.OccurrencesFinder;
import kenya.eclipse.ast.SimpleASTProvider;
import kenya.eclipse.ast.bindings.BuiltinMethodBinding;
import kenya.eclipse.ast.bindings.ClassBinding;
import kenya.eclipse.ast.bindings.EnumBinding;
import kenya.eclipse.ast.bindings.EnumConstantBinding;
import kenya.eclipse.multieditor.kenya.KenyaEditor;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentModificationOperation;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentTextOperation;
import kenya.eclipse.multieditor.kenya.util.LocationUtils;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.ICheckedCode;
import minijava.node.Start;
import minijava.node.TIdentifier;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.ui.IWorkbenchSite;

/**
 * Renames a Java element or workbench resource.
 * <p>
 * Action is applicable to selections containing elements of type
 * <code>IJavaElement</code> or <code>IResource</code>.
 * 
 * <p>
 * This class may be instantiated; it is not intended to be subclassed.
 * </p>
 * 
 * @since 2.0
 */
public class RenameAction extends SelectionDispatchAction {

	private OccurrencesFinder fOccurrenceFinder;
	private KenyaEditor fEditor;
	
	
	/**
	 * Creates a new <code>RenameAction</code>. The action requires
	 * that the selection provided by the site's selection provider is of type <code>
	 * org.eclipse.jface.viewers.IStructuredSelection</code>.
	 * 
	 * @param site the site providing context information for this action
	 */
	public RenameAction(IWorkbenchSite site) {
		super(site);
		setText("Rename"); //$NON-NLS-1$
		fOccurrenceFinder = new OccurrencesFinder();
//		WorkbenchHelp.setHelp(this, IKenyaHelpContextIds.RENAME_ACTION);
	}
	
	/**
	 * Note: This constructor is for internal use only. Clients should not call this constructor.
	 */
	public RenameAction(KenyaEditor editor) {
		this(editor.getEditorSite());
		fEditor= editor;
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.ui.actions.SelectionDispatchAction#selectionChanged(org.eclipse.jface.viewers.ISelection)
	 */
	public void selectionChanged(ISelection selection) {
		if(selection instanceof ITextSelection) {
			selectionChanged((ITextSelection)selection);
		}
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.ui.actions.SelectionDispatchAction#update(org.eclipse.jface.viewers.ISelection)
	 */
	public void update(ISelection selection) {
		super.update(selection);
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.ui.actions.SelectionDispatchAction#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event) {
		super.selectionChanged(event);
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.ui.actions.SelectionDispatchAction#selectionChanged(org.eclipse.jface.text.ITextSelection)
	 */
	public void selectionChanged(ITextSelection selection) {
		try {
			IDocument doc = fEditor.getDocumentProvider().getDocument(fEditor.getEditorInput());
			
			setEnabled(computeEnabledState(doc, selection.getOffset(), selection.getLength()));
			
		} catch(NullPointerException e) {
			setEnabled(false);
		}
	}
	
	private boolean computeEnabledState(IDocument doc, int offset, int length){
		if(fEditor!=null) {
			ICheckedCode currentcode = fEditor.getKenyaCodeManager().getLastCheckedCode();
			if(currentcode==null) return false;
			if(!currentcode.isErroredCode()) {
				Start start = SimpleASTProvider.getAST(currentcode);
				String msg
				  = fOccurrenceFinder.initialize(start, currentcode, doc, offset, length);
				IBinding tgt = fOccurrenceFinder.getTarget();
				return msg==null &&
				//the following are not renamable
					!(   tgt instanceof EnumBinding
					  || tgt instanceof EnumConstantBinding
						|| tgt instanceof BuiltinMethodBinding);
			}
		}
		return false;
	}
	
	public void run(IStructuredSelection selection) {
//		 if (fRenameJavaElement.isEnabled())
//			fRenameJavaElement.run(selection);
//		if (fRenameResource != null && fRenameResource.isEnabled())
//			fRenameResource.run(selection);
	}
	
	public void run(ITextSelection selection) {
		selectionChanged(selection);
		if(!isEnabled()) return;
		
		IDocument doc = fEditor.getDocumentProvider().getDocument(fEditor.getEditorInput());
		if(doc==null) return;
		
		List list = fOccurrenceFinder.perform();
		
		IBinding tgt = fOccurrenceFinder.getTarget();
		String initial = fOccurrenceFinder.getExpandedSelectionText();
		
		//ask the user to specify a new name
		InputDialog d
		  = new InputDialog(KenyaPlugin.getActiveWorkbenchShell(),
		  		"Rename..", "Please enter a new identifier.",
					initial,
					new Validator(tgt)
				);
		d.create();
		int choice = d.open();
		if(choice != Dialog.OK) return;
		
		String newId = d.getValue();
		
		if(newId.equals(initial)) return;
		
		DocumentModificationOperation op = new DocumentModificationOperation();
		
		for(Iterator iter = list.iterator(); iter.hasNext();) {
			TIdentifier tid = (TIdentifier)iter.next();
			
			String n = tid.getText();
			if(!n.equals(initial)) continue;
			
			//calculate the offset and length
			ISourceCodeLocation loc
			  = AdvancedPositionFinder.getFullLocation(tid, doc);
			Position pos = LocationUtils.convert(loc, doc);
			
			DocumentTextOperation dto
			  = DocumentTextOperation.newTextReplacement(pos.offset, pos.length, newId);
			op.addOperation(dto);
			
		}
		
		op.run(doc);
	}
}

class Validator implements IInputValidator {
	
	private IBinding tgt;
	
	public Validator(IBinding target) {
		tgt = target;
	}
	
	public String isValid(String txt) {
		if(txt==null || txt.length() == 0 ) {
			return "Name is empty";
		} else if(KenyaConventions.clashesWithReservedWord(txt)) {
			return "Cannot give the same name as a reserved keyword.";
		} else if(tgt instanceof ClassBinding) {
			IStatus st = KenyaConventions.validateKenyaTypeName(txt);
			switch(st.getSeverity()) {
				case IStatus.OK: return null;
				default: return st.getMessage();
			}
		} else {
			char[] cs = txt.toCharArray();
			boolean valid = Character.isJavaIdentifierStart(cs[0]) //txt has at least length 1
			  && !Character.isDigit(cs[0]); 
			if(!valid) {
				return "Identifiers must start with an underscore or letter.";
			}
			for(int i = 1; i < cs.length; i++) {
				if(!valid) {
					return "Identifiers cannot contain spaces or other special characters.";
				}
				valid = Character.isJavaIdentifierPart(cs[i]);
			}
			return null;
		}
	}
}


