package kenya.eclipse.ui.wizards;

import kenya.eclipse.KenyaConstants;
import kenya.eclipse.KenyaConventions;
import kenya.eclipse.ui.wizards.dialogfields.DialogField;
import kenya.eclipse.ui.wizards.dialogfields.LayoutUtil;
import kenya.eclipse.ui.wizards.dialogfields.SelectionButtonDialogFieldGroup;
import kenya.eclipse.ui.wizards.dialogfields.Separator;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.IDialogPage;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ContainerSelectionDialog;

/**
 * The "New" wizard page allows setting the container for
 * the new file as well as the file name. The page
 * will only accept file name without the extension OR
 * with the extension that matches the expected one (k).
 */

public class NewKFileWizardPage extends WizardPage {
	private Text containerText;
	private Text fileText;
	private ISelection selection;
	private SelectionButtonDialogFieldGroup fMethodStubsButtons;
	
	public static final int NO_MAIN = 0;
	public static final int CREATE_MAIN = 1;
	public static final int CREATE_MAIN_ARGS = 2;
	
	/**
	 * Constructor for SampleNewWizardPage.
	 * @param pageName
	 */
	public NewKFileWizardPage(ISelection selection) {
		super("wizardPage");
		setTitle("Kenya File");
		setDescription("Create a new Kenya file (a 'class' in Java)");
		this.selection = selection;
		
		//this has to match the constants above in terms of order and index
		String[] buttonNames3= new String[] {
				"none",
				"void main()",
				"void main(String[] args)"
		};
		fMethodStubsButtons= new SelectionButtonDialogFieldGroup(SWT.RADIO, buttonNames3, 1);
		fMethodStubsButtons.setLabelText("Would you like to generate a main method?");
	}
	
	/**
	 * @see IDialogPage#createControl(Composite)
	 */
	public void createControl(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout();
		container.setLayout(layout);
		layout.numColumns = 3;
		layout.verticalSpacing = 9;
		Label label = new Label(container, SWT.NULL);
		label.setText("&Container:");
		
		containerText = new Text(container, SWT.BORDER | SWT.SINGLE);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		containerText.setLayoutData(gd);
		containerText.setEditable(false); //make sure of legal input
		containerText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				dialogChanged();
			}
		});
		
		Button button = new Button(container, SWT.PUSH);
		button.setText("Browse...");
		button.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleBrowse();
			}
		});
		label = new Label(container, SWT.NULL);
		label.setText("&Type name:");
		
		fileText = new Text(container, SWT.BORDER | SWT.SINGLE);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		fileText.setLayoutData(gd);
		fileText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				dialogChanged();
			}
		});
		DialogField.createEmptySpace(container);
		
		createSeparator(container, layout.numColumns);
		createMethodStubSelectionControls(container, layout.numColumns);
		
		initialize();
		dialogChanged();
		setControl(container);
	}
	
	private void createMethodStubSelectionControls(Composite composite, int nColumns) {
		Control labelControl = fMethodStubsButtons.getLabelControl(composite);
		LayoutUtil.setHorizontalSpan(labelControl, nColumns);
		
		DialogField.createEmptySpace(composite);
		
		Control buttonGroup = fMethodStubsButtons.getSelectionButtonsGroup(composite);
		LayoutUtil.setHorizontalSpan(buttonGroup, nColumns - 1);
	}
	
	/**
	 * Creates a separator line. Expects a <code>GridLayout</code> with at least 1 column.
	 * 
	 * @param composite the parent composite
	 * @param nColumns number of columns to span
	 */
	protected void createSeparator(Composite composite, int nColumns) {
		(new Separator(SWT.SEPARATOR | SWT.HORIZONTAL)).doFillIntoGrid(composite, nColumns, convertHeightInCharsToPixels(1));		
	}
	
	/**
	 * Tests if the current workbench selection is a suitable
	 * container to use.
	 */
	private void initialize() {
		if (selection!=null && selection.isEmpty()==false && selection instanceof IStructuredSelection) {
			IStructuredSelection ssel = (IStructuredSelection)selection;
			if (ssel.size()>1) return;
			Object obj = ssel.getFirstElement();
			if (obj instanceof IResource) {
				IContainer container;
				if (obj instanceof IContainer)
					container = (IContainer)obj;
				else
					container = ((IResource)obj).getParent();
				containerText.setText(container.getFullPath().toString());
			}
		}
		fileText.setText("Exercise");
		fMethodStubsButtons.setSelection(CREATE_MAIN, true);
	}
	
	/**
	 * Uses the standard container selection dialog to
	 * choose the new value for the container field.
	 */
	private void handleBrowse() {
		ContainerSelectionDialog dialog =
			new ContainerSelectionDialog(
					getShell(),
					ResourcesPlugin.getWorkspace().getRoot(),
					false,
			"Select new file container");
		if (dialog.open() == ContainerSelectionDialog.OK) {
			Object[] result = dialog.getResult();
			if (result.length == 1) {
				containerText.setText(((Path)result[0]).toOSString());
			}
		}
	}
	
	/**
	 * Ensures that both text fields are set.
	 */
	private void dialogChanged() {
		String container = getContainerName();
		String typeName = getFileName();
		
		if(container==null || container.length()==0) {
			updateStatus("Container is empty", ERROR);
			return;
		}
		
		Path file = new Path(container+Path.SEPARATOR+typeName+KenyaConstants.SUFFIX_STRING_k);
		if(ResourcesPlugin.getWorkspace().getRoot().exists(file)) {
			updateStatus("A file with the same name already exists at the specified location", ERROR);
			return;
		}
		
		IStatus status = KenyaConventions.validateKenyaTypeName(typeName);
		switch(status.getSeverity()) {
			case IStatus.ERROR:
				updateStatus(status.getMessage(), ERROR);
				return;
			case IStatus.WARNING:
				updateStatus(status.getMessage(), WARNING);
				return;
			default:
				updateStatus(null, NONE);
				return;
		}
	}
	
	protected void updateStatus(String message, int type) {
		setMessage(message, type);
		setPageComplete(type!=ERROR);
	}
	
	public String getContainerName() {
		return containerText.getText();
	}
	public String getFileName() {
		return fileText.getText();
	}
	
	public int getMainMethodCreationType() {
		if(fMethodStubsButtons.isSelected(NO_MAIN)) {
			return NO_MAIN;
		} else if(fMethodStubsButtons.isSelected(CREATE_MAIN)) {
			return CREATE_MAIN;
		} else if(fMethodStubsButtons.isSelected(CREATE_MAIN_ARGS)) {
			return CREATE_MAIN_ARGS;
		} else {
			return -1;
		}
	}
	
}