package kenya.eclipse;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import kenya.eclipse.buildext.IKenyaPostBuildAnalyserFactory;
import kenya.eclipse.debug.KenyaRunner;
import kenya.eclipse.multieditor.java.util.JavaCodeScanner;
import kenya.eclipse.multieditor.kenya.KenyaPartitionScanner;
import kenya.eclipse.multieditor.kenya.util.KenyaCodeScanner;
import kenya.eclipse.multieditor.util.DefaultSyntaxColorProvider;
import kenya.eclipse.multieditor.util.KenyaColourManager;
import kenya.eclipse.preferences.PreferenceConstants;
import mediator.subscription.MediationService;

import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.rules.RuleBasedScanner;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.editors.text.EditorsUI;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.eclipse.ui.texteditor.ChainedPreferenceStore;
import org.osgi.framework.BundleContext;

/**
 * The main plugin class to be used in the desktop.
 */
public class KenyaPlugin extends AbstractUIPlugin {
	
	//The shared instance.
	private static KenyaPlugin fgInstance;

	//Resource bundle.
	private ResourceBundle resourceBundle;
	
	/**
	 * combined preference store
	 */
	private IPreferenceStore fCombinedPreferenceStore;
	
	private DefaultSyntaxColorProvider fColorProvider;
	
	private KenyaPartitionScanner fPartitionScanner;
	private KenyaCodeScanner fCodeScanner;
	private JavaCodeScanner fjCodeScanner;
	
	private KenyaRunner fKenyaRunner;
	

	// Map of PostBuildAnalyser IDs to IKenyaPostBuildAnalysers
	protected Map fPostBuildAnalyserMap;
	
	/**
	 * the ColourManager
	 */
	private KenyaColourManager fColourManager;
	
	
	/**
	 * The constructor.
	 */
	public KenyaPlugin() {
		super();
		fgInstance = this;
		try {
			resourceBundle = ResourceBundle.getBundle("kenya.eclipse.KenyaPluginResources");
		} catch (MissingResourceException x) {
			resourceBundle = null;
		}
	}
	
	/**
	 * This method is called upon plug-in activation
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
		MediationService.getInstance().start();
	}
	
	/**
	 * This method is called when the plug-in is stopped
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
		MediationService.getInstance().shutdown();
	}
	
	/**
	 * Returns the shared instance.
	 */
	public static KenyaPlugin getDefault() {
		return fgInstance;
	}
	
	/**
	 * Returns a combined preference store, this store is read-only.
	 * @return the combined preference store
	 */
	public IPreferenceStore getCombinedPreferenceStore() {
		if (fCombinedPreferenceStore == null) {
			IPreferenceStore generalTextStore= EditorsUI.getPreferenceStore(); 
			fCombinedPreferenceStore = new ChainedPreferenceStore( new IPreferenceStore[] { getPreferenceStore(), generalTextStore });
		}
		return fCombinedPreferenceStore;
	}
	
	/*
	 *  (non-Javadoc)
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#initializeDefaultPreferences(org.eclipse.jface.preference.IPreferenceStore)
	 */
	protected void initializeDefaultPreferences(IPreferenceStore store) {
		EditorsUI.useAnnotationsPreferencePage(store);
		EditorsUI.useQuickDiffPreferencePage(store);
		PreferenceConstants.initializeDefaultValues(store);
	}
	
	/**
	 * Return a scanner for creating kenya partitions.
	 */
	public KenyaPartitionScanner getKenyaPartitionScanner() {
		if (fPartitionScanner == null)
			fPartitionScanner= new KenyaPartitionScanner();
		return fPartitionScanner;
	}
	
	/**
	 * Returns the kenya singleton scanner.
	 */
	public RuleBasedScanner getKenyaCodeScanner() {
		if (fCodeScanner == null)
			fCodeScanner= new KenyaCodeScanner(getSyntaxColorProvider());
		return fCodeScanner;
	}
	
	/**
	 * Returns a shared instance of the kenya default color provider.
	 */
	public DefaultSyntaxColorProvider getSyntaxColorProvider() {
		if (fColorProvider == null)
			fColorProvider= new DefaultSyntaxColorProvider();
		return fColorProvider;
	}
	
	/**
	 * Returns the java singleton scanner.
	 */
	public RuleBasedScanner getJavaCodeScanner() {
		if (fjCodeScanner == null)
			fjCodeScanner= new JavaCodeScanner(getSyntaxColorProvider());
		return fjCodeScanner;
	}
	
	/**
	 * Returns the workspace instance.
	 */
	public static IWorkspace getWorkspace() {
		return ResourcesPlugin.getWorkspace();
	}
	
	/**
	 * Returns the active workbench window
	 * 
	 * @return the active workbench window
	 */
	public static IWorkbenchWindow getActiveWorkbenchWindow() {
		return getDefault().getWorkbench().getActiveWorkbenchWindow();
	}	
	
	public static IWorkbenchPage getActivePage() {
		IWorkbenchWindow w = getActiveWorkbenchWindow();
		if (w != null) {
			return w.getActivePage();
		}
		return null;
	}
	
	/**
	 * Returns the active workbench shell or <code>null</code> if none
	 * 
	 * @return the active workbench shell or <code>null</code> if none
	 */
	public static Shell getActiveWorkbenchShell() {
		IWorkbenchWindow window = getActiveWorkbenchWindow();
		if (window != null) {
			return window.getShell();
		}
		return null;
	}	
	
	
	public static String getPluginId() {
		return KenyaConstants.K_PLUGIN_ID;
	}
	
	public static void log(IStatus status) {
		getDefault().getLog().log(status);
	}
	
	public static void log(String message) {
		log(createErrorStatus(message, null));
	}
	
	public static void log(String message, Throwable exception) {
		log(createErrorStatus(message, exception));
	}
	
	public static void log(Throwable e) {
		log(createErrorStatus(null, e));
	}
	
	private static IStatus createErrorStatus(
			String message,
			Throwable exception)
	{
		if(message == null) {
			message = "Internal Error";
		}
		return new Status(
				IStatus.ERROR,
				getPluginId(),
				IStatus.ERROR,
				message,
				exception
		);
	}
	
	public static void errorDialog(String message, IStatus status) {
		log(status);
		Shell shell = getActiveWorkbenchShell();
		if (shell != null) {
			ErrorDialog.openError(shell, "Execution error:", message, status); //$NON-NLS-1$
		}
	}
	
	/**
	 * Utility method with conventions
	 */
	public static void errorDialog(String message, Throwable t) {
		log(t);
		Shell shell = getActiveWorkbenchShell();
		if (shell != null) {
			IStatus status = createErrorStatus("Error logged from Kenya Debugging: ", t);
			ErrorDialog.openError(shell, "Execution error:", message, status);
		}
	}
	
	/**
	 * 
	 */
	public KenyaRunner getKenyaRunner() {
		if(fKenyaRunner==null) {
			fKenyaRunner = new KenyaRunner();
		}
		return fKenyaRunner;
	}
	
	
	/**
	 * Return a collection of all registered <code>IKenyaPostBuildAnalyser</code>s
	 */
	public Collection getPostBuildAnalyserFactories() {
		if (fPostBuildAnalyserMap == null) {
			initializePostBuildAnalyserMap();
		}
		Collection c = fPostBuildAnalyserMap.values();
		
		ArrayList l = new ArrayList(c.size());
		for(Iterator iter = c.iterator(); iter.hasNext();) {
			IConfigurationElement element = (IConfigurationElement)iter.next();
			try {
				l.add((IKenyaPostBuildAnalyserFactory)element.createExecutableExtension("class"));
			} catch(CoreException e) {
				//ignore, try next
			}
		}
		return l;
	}
	
	protected void initializePostBuildAnalyserMap() {
		fPostBuildAnalyserMap = new HashMap(10);
		
		IExtensionPoint extensionPoint= Platform.getExtensionRegistry().getExtensionPoint(getPluginId(), KenyaConstants.EXTENSION_POINT_BUILD_PARTICIPANT);
		IConfigurationElement[] infos= extensionPoint.getConfigurationElements();
		for (int i = 0; i < infos.length; i++) {
			String id = infos[i].getAttribute("id");
			fPostBuildAnalyserMap.put(id, infos[i]);
		}
	}
	

	/**
	 * Returns the plugin's resource bundle,
	 */
	public ResourceBundle getResourceBundle() {
		return resourceBundle;
	}

	/**
	 * @return
	 */
	public KenyaColourManager getKenyaColourManager() {
		if (fColourManager == null) {
			fColourManager = new KenyaColourManager();
		}

		return fColourManager;
	}
	
}
