/*
 * Created on 25-Feb-2005
 */
package kenya.eclipse.ast;

import kenya.eclipse.multieditor.kenya.util.LocationUtils;
import kenya.eclipse.multieditor.kenya.util.SourceCodeLocationComparator;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.*;
import minijava.node.Node;
import minijava.node.Start;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;

/**
 * @author Thomas Timbul
 */
public class NodeFinder extends DepthFirstAdapter {
	
	private int fStart;
	private int fEnd;
	private IDocument fDocument;
	
	private Node fCoveringNode;
	private Node fCoveredNode;
	
	protected NodeFinder(ISourceCodeLocation loc, IDocument doc) {
		this(LocationUtils.convert(loc, doc), doc);
	}
	
	protected NodeFinder(Position p, IDocument doc) {
		fDocument = doc;
		fStart = p.getOffset();
		fEnd = fStart + p.getLength();
	}
	
	/**
	 * finds the Node at the given location in the given AST and document
	 */
	public static Node perform(Start root, IDocument doc, ISourceCodeLocation location) {
		NodeFinder finder= new NodeFinder(location, doc);
		root.apply(finder);
		Node result= finder.getCoveredNode();
		if (result == null) {
			return finder.getCoveringNode();
		}
		//is what we found an exact match?
		ISourceCodeLocation foundLoc = AdvancedPositionFinder.getFullLocation(result, null);
		int comparison = SourceCodeLocationComparator.getInstance().compare(foundLoc, location);
		if(comparison!=0) {
			return finder.getCoveringNode();
		}
		return result;
	}
	
	/**
	 * finds the Node at the given Position in the given document
	 */
	public static Node perform(Start root, IDocument doc, Position pos) {
		return perform(root, doc, LocationUtils.convert(pos, doc));
	}
	
	public void defaultCase(Node node) {
		if(isFlabby(node)) super.defaultCase(node);
	}
	
	/**
	 * Returns the covered node. If more than one nodes are covered by the selection, the
	 * returned node is first covered node found in a top-down traversal of the AST
	 * @return Node
	 */
	public Node getCoveredNode() {
		return fCoveredNode;
	}
	
	/**
	 * Returns the covering node. If more than one nodes are covering the selection, the
	 * returned node is last covering node found in a top-down traversal of the AST
	 * @return Node
	 */
	public Node getCoveringNode() {
		return fCoveringNode;
	}
	
	private boolean isFlabby(Node node) {
		if(!(node instanceof Token)) return true;
		boolean flabby = true; //flabby defines whether we should traverse further down
		try {
			ISourceCodeLocation loc = AdvancedPositionFinder.getFullLocation(node, fDocument);
			Position pos = LocationUtils.convert(loc, fDocument);
			
			int nodeStart = pos.getOffset();
			int nodeEnd = nodeStart + pos.getLength();
			
			//no overlap
			if (nodeEnd < fStart || fEnd < nodeStart) {
				flabby = false;
			}
			
			// this node covers the given position
			if (nodeStart <= fStart && fEnd <= nodeEnd) {
				fCoveringNode = node;
			}
			
			// the given position covers this node
			if (fStart <= nodeStart && nodeEnd <= fEnd) {
				if (fCoveringNode == node) { // nodeStart == fStart && nodeEnd == fEnd
					fCoveredNode = node;
					flabby = true; // look further for node with same length as parent
				} else if (fCoveredNode == null) { // no better found
					fCoveredNode = node;
				}
				flabby = false;
			}
		} catch(NullPointerException e) {
			flabby = false;
		}
		return flabby;
	}
	
	public void caseAAndBoolTerm(AAndBoolTerm node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAAndBoolTerm(node);
	}
	public void caseAArrayAccess(AArrayAccess node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAArrayAccess(node);
	}
	public void caseAArrayAllocate(AArrayAllocate node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAArrayAllocate(node);
	}
	public void caseAArrayCommaInitList(AArrayCommaInitList node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAArrayCommaInitList(node);
	}
	public void caseAArrayDecInnerDeclaration(AArrayDecInnerDeclaration node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAArrayDecInnerDeclaration(node);
	}
	public void caseAArrayFieldAccess(AArrayFieldAccess node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAArrayFieldAccess(node);
	}
	public void caseAArrayInit(AArrayInit node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAArrayInit(node);
	}
	public void caseAArrayInitList(AArrayInitList node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAArrayInitList(node);
	}
	public void caseAArrayIntExpression(AArrayIntExpression node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAArrayIntExpression(node);
	}
	public void caseAAssertStatement(AAssertStatement node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAAssertStatement(node);
	}
	public void caseAAssignment(AAssignment node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAAssignment(node);
	}
	public void caseAAssignmentForLeftStat(AAssignmentForLeftStat node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAAssignmentForLeftStat(node);
	}
	public void caseAAssignmentForRightStat(AAssignmentForRightStat node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAAssignmentForRightStat(node);
	}
	public void caseAAssignmentStatement(AAssignmentStatement node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAAssignmentStatement(node);
	}
	public void caseABasicTypeType(ABasicTypeType node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseABasicTypeType(node);
	}
	public void caseABlock(ABlock node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseABlock(node);
	}
	public void caseABlockElseFollow(ABlockElseFollow node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseABlockElseFollow(node);
	}
	public void caseABlockStatement(ABlockStatement node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseABlockStatement(node);
	}
	public void caseABooleanBasicType(ABooleanBasicType node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseABooleanBasicType(node);
	}
	public void caseABoolexpExpression(ABoolexpExpression node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseABoolexpExpression(node);
	}
	public void caseABoolliteralFactor(ABoolliteralFactor node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseABoolliteralFactor(node);
	}
	public void caseABreakStatement(ABreakStatement node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseABreakStatement(node);
	}
	public void caseACasePossibleCase(ACasePossibleCase node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseACasePossibleCase(node);
	}
	public void caseACharBasicType(ACharBasicType node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseACharBasicType(node);
	}
	public void caseACharliteralFactor(ACharliteralFactor node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseACharliteralFactor(node);
	}
	public void caseAClassDecDeclaration(AClassDecDeclaration node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAClassDecDeclaration(node);
	}
	public void caseAClassInnerDeclaration(AClassInnerDeclaration node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAClassInnerDeclaration(node);
	}
	public void caseAClassTypeReferenceType(AClassTypeReferenceType node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAClassTypeReferenceType(node);
	}
	public void caseAColonString(AColonString node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseAColonString(node);
	}
	public void caseACommaArrayInit(ACommaArrayInit node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseACommaArrayInit(node);
	}
	public void caseACommaEnumList(ACommaEnumList node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseACommaEnumList(node);
	}
	public void caseACommaExp(ACommaExp node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseACommaExp(node);
	}
	public void caseACommaType(ACommaType node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseACommaType(node);
	}
	public void caseACommaTypeName(ACommaTypeName node) {
		if(isFlabby(node)) if(isFlabby(node)) super.caseACommaTypeName(node);
	}
	public void caseAConstDecDeclaration(AConstDecDeclaration node) {
		if(isFlabby(node)) super.caseAConstDecDeclaration(node);
	}
	public void caseADefaultPossibleCase(ADefaultPossibleCase node) {
		if(isFlabby(node)) super.caseADefaultPossibleCase(node);
	}
	public void caseADivTerm(ADivTerm node) {
		if(isFlabby(node)) super.caseADivTerm(node);
	}
	public void caseADNumber(ADNumber node) {
		if(isFlabby(node)) super.caseADNumber(node);
	}
	public void caseADoubleBasicType(ADoubleBasicType node) {
		if(isFlabby(node)) super.caseADoubleBasicType(node);
	}
	public void caseADynamicArrayInitialiser(ADynamicArrayInitialiser node) {
		if(isFlabby(node)) super.caseADynamicArrayInitialiser(node);
	}
	public void caseAElsePart(AElsePart node) {
		if(isFlabby(node)) super.caseAElsePart(node);
	}
	public void caseAEmptyDeclarations(AEmptyDeclarations node) {
		super.caseAEmptyDeclarations(node);
	}
	public void caseAEmptyStatements(AEmptyStatements node) {
		super.caseAEmptyStatements(node);
	}
	public void caseAEnumDecDeclaration(AEnumDecDeclaration node) {
		if(isFlabby(node)) super.caseAEnumDecDeclaration(node);
	}
	public void caseAEnumList(AEnumList node) {
		if(isFlabby(node)) super.caseAEnumList(node);
	}
	public void caseAEqEquality(AEqEquality node) {
		if(isFlabby(node)) super.caseAEqEquality(node);
	}
	public void caseAEqualityBoolTerm(AEqualityBoolTerm node) {
		if(isFlabby(node)) super.caseAEqualityBoolTerm(node);
	}
	public void caseAExpActualParamList(AExpActualParamList node) {
		if(isFlabby(node)) super.caseAExpActualParamList(node);
	}
	public void caseAExpressionFactor(AExpressionFactor node) {
		if(isFlabby(node)) super.caseAExpressionFactor(node);
	}
	public void caseAFactorPostfixExp(AFactorPostfixExp node) {
		if(isFlabby(node)) super.caseAFactorPostfixExp(node);
	}
	public void caseAFalseBooleanliteral(AFalseBooleanliteral node) {
		if(isFlabby(node)) super.caseAFalseBooleanliteral(node);
	}
	public void caseAFieldaccessFactor(AFieldaccessFactor node) {
		if(isFlabby(node)) super.caseAFieldaccessFactor(node);
	}
	public void caseAFormalParamList(AFormalParamList node) {
		if(isFlabby(node)) super.caseAFormalParamList(node);
	}
	public void caseAForStatement(AForStatement node) {
		if(isFlabby(node)) super.caseAForStatement(node);
	}
	public void caseAForUnaryStatement(AForUnaryStatement node) {
		if(isFlabby(node)) super.caseAForUnaryStatement(node);
	}
	public void caseAFuncDecDeclaration(AFuncDecDeclaration node) {
		if(isFlabby(node)) super.caseAFuncDecDeclaration(node);
	}
	public void caseAFunctionApplication(AFunctionApplication node) {
		if(isFlabby(node)) super.caseAFunctionApplication(node);
	}
	public void caseAFunctioncallForRightStat(AFunctioncallForRightStat node) {
		if(isFlabby(node)) super.caseAFunctioncallForRightStat(node);
	}
	public void caseAFunctioncallStatement(AFunctioncallStatement node) {
		if(isFlabby(node)) super.caseAFunctioncallStatement(node);
	}
	public void caseAFunctionFactor(AFunctionFactor node) {
		if(isFlabby(node)) super.caseAFunctionFactor(node);
	}
	public void caseAGteqRelational(AGteqRelational node) {
		if(isFlabby(node)) super.caseAGteqRelational(node);
	}
	public void caseAGtRelational(AGtRelational node) {
		if(isFlabby(node)) super.caseAGtRelational(node);
	}
	public void caseAIfElseFollow(AIfElseFollow node) {
		if(isFlabby(node)) super.caseAIfElseFollow(node);
	}
	public void caseAIfStat(AIfStat node) {
		if(isFlabby(node)) super.caseAIfStat(node);
	}
	public void caseAIfStatement(AIfStatement node) {
		if(isFlabby(node)) super.caseAIfStatement(node);
	}
	public void caseAInitialiser(AInitialiser node) {
		if(isFlabby(node)) super.caseAInitialiser(node);
	}
	public void caseAInnerDecForLeftStat(AInnerDecForLeftStat node) {
		if(isFlabby(node)) super.caseAInnerDecForLeftStat(node);
	}
	public void caseAInnerDecStatement(AInnerDecStatement node) {
		if(isFlabby(node)) super.caseAInnerDecStatement(node);
	}
	public void caseAIntBasicType(AIntBasicType node) {
		if(isFlabby(node)) super.caseAIntBasicType(node);
	}
	public void caseAINumber(AINumber node) {
		if(isFlabby(node)) super.caseAINumber(node);
	}
	public void caseAJavaForControl(AJavaForControl node) {
		if(isFlabby(node)) super.caseAJavaForControl(node);
	}
	public void caseAListActualParamList(AListActualParamList node) {
		if(isFlabby(node)) super.caseAListActualParamList(node);
	}
	public void caseAListDeclarations(AListDeclarations node) {
		if(isFlabby(node)) super.caseAListDeclarations(node);
	}
	public void caseAListStatements(AListStatements node) {
		if(isFlabby(node)) super.caseAListStatements(node);
	}
	public void caseALteqRelational(ALteqRelational node) {
		if(isFlabby(node)) super.caseALteqRelational(node);
	}
	public void caseALtRelational(ALtRelational node) {
		if(isFlabby(node)) super.caseALtRelational(node);
	}
	public void caseAMathRelational(AMathRelational node) {
		if(isFlabby(node)) super.caseAMathRelational(node);
	}
	public void caseAMinusMathExpression(AMinusMathExpression node) {
		if(isFlabby(node)) super.caseAMinusMathExpression(node);
	}
	public void caseAMinusUnaryExp(AMinusUnaryExp node) {
		if(isFlabby(node)) super.caseAMinusUnaryExp(node);
	}
	public void caseAModTerm(AModTerm node) {
		if(isFlabby(node)) super.caseAModTerm(node);
	}
	public void caseAMultTerm(AMultTerm node) {
		if(isFlabby(node)) super.caseAMultTerm(node);
	}
	public void caseANegateUnaryExp(ANegateUnaryExp node) {
		if(isFlabby(node)) super.caseANegateUnaryExp(node);
	}
	public void caseANeqEquality(ANeqEquality node) {
		if(isFlabby(node)) super.caseANeqEquality(node);
	}
	public void caseANormalArrayInitialiser(ANormalArrayInitialiser node) {
		if(isFlabby(node)) super.caseANormalArrayInitialiser(node);
	}
	public void caseANullFactor(ANullFactor node) {
		if(isFlabby(node)) super.caseANullFactor(node);
	}
	public void caseANumberFactor(ANumberFactor node) {
		if(isFlabby(node)) super.caseANumberFactor(node);
	}
	public void caseAOrBoolExpression(AOrBoolExpression node) {
		if(isFlabby(node)) super.caseAOrBoolExpression(node);
	}
	public void caseAPlusMathExpression(APlusMathExpression node) {
		if(isFlabby(node)) super.caseAPlusMathExpression(node);
	}
	public void caseAPlusUnaryExp(APlusUnaryExp node) {
		if(isFlabby(node)) super.caseAPlusUnaryExp(node);
	}
	public void caseAPostdecr(APostdecr node) {
		if(isFlabby(node)) super.caseAPostdecr(node);
	}
	public void caseAPostdecrForUnaryExp(APostdecrForUnaryExp node) {
		if(isFlabby(node)) super.caseAPostdecrForUnaryExp(node);
	}
	public void caseAPostfixExpUnaryExp(APostfixExpUnaryExp node) {
		if(isFlabby(node)) super.caseAPostfixExpUnaryExp(node);
	}
	public void caseAPostincr(APostincr node) {
		if(isFlabby(node)) super.caseAPostincr(node);
	}
	public void caseAPostincrForUnaryExp(APostincrForUnaryExp node) {
		if(isFlabby(node)) super.caseAPostincrForUnaryExp(node);
	}
	public void caseAPredecr(APredecr node) {
		if(isFlabby(node)) super.caseAPredecr(node);
	}
	public void caseAPredecrForUnaryExp(APredecrForUnaryExp node) {
		if(isFlabby(node)) super.caseAPredecrForUnaryExp(node);
	}
	public void caseAPreincr(APreincr node) {
		if(isFlabby(node)) super.caseAPreincr(node);
	}
	public void caseAPreincrForUnaryExp(APreincrForUnaryExp node) {
		if(isFlabby(node)) super.caseAPreincrForUnaryExp(node);
	}
	public void caseAQualifiedName(AQualifiedName node) {
		if(isFlabby(node)) super.caseAQualifiedName(node);
	}
	public void caseAQualifiedNameName(AQualifiedNameName node) {
		if(isFlabby(node)) super.caseAQualifiedNameName(node);
	}
	public void caseAReferenceTypeType(AReferenceTypeType node) {
		if(isFlabby(node)) super.caseAReferenceTypeType(node);
	}
	public void caseARelationalEquality(ARelationalEquality node) {
		if(isFlabby(node)) super.caseARelationalEquality(node);
	}
	public void caseAReturnStatement(AReturnStatement node) {
		if(isFlabby(node)) super.caseAReturnStatement(node);
	}
	public void caseAScalarCommaInitList(AScalarCommaInitList node) {
		if(isFlabby(node)) super.caseAScalarCommaInitList(node);
	}
	public void caseAScalarFieldAccess(AScalarFieldAccess node) {
		if(isFlabby(node)) super.caseAScalarFieldAccess(node);
	}
	public void caseAScalarInitList(AScalarInitList node) {
		if(isFlabby(node)) super.caseAScalarInitList(node);
	}
	public void caseASimpleName(ASimpleName node) {
		if(isFlabby(node)) super.caseASimpleName(node);
	}
	public void caseASimpleNameName(ASimpleNameName node) {
		if(isFlabby(node)) super.caseASimpleNameName(node);
	}
	public void caseAStaticArrayInitialiser(AStaticArrayInitialiser node) {
		if(isFlabby(node)) super.caseAStaticArrayInitialiser(node);
	}
	public void caseAStringBasicType(AStringBasicType node) {
		if(isFlabby(node)) super.caseAStringBasicType(node);
	}
	public void caseAStringliteralFactor(AStringliteralFactor node) {
		if(isFlabby(node)) super.caseAStringliteralFactor(node);
	}
	public void caseASwitchBlock(ASwitchBlock node) {
		if(isFlabby(node)) super.caseASwitchBlock(node);
	}
	public void caseASwitchStatement(ASwitchStatement node) {
		if(isFlabby(node)) super.caseASwitchStatement(node);
	}
	public void caseATermBoolExpression(ATermBoolExpression node) {
		if(isFlabby(node)) super.caseATermBoolExpression(node);
	}
	public void caseATermMathExpression(ATermMathExpression node) {
		if(isFlabby(node)) super.caseATermMathExpression(node);
	}
	public void caseATrueBooleanliteral(ATrueBooleanliteral node) {
		if(isFlabby(node)) super.caseATrueBooleanliteral(node);
	}
	public void caseATypeName(ATypeName node) {
		if(isFlabby(node)) super.caseATypeName(node);
	}
	public void caseATypeParam(ATypeParam node) {
		if(isFlabby(node)) super.caseATypeParam(node);
	}
	public void caseATypeParamList(ATypeParamList node) {
		if(isFlabby(node)) super.caseATypeParamList(node);
	}
	public void caseAUnaryForRightStat(AUnaryForRightStat node) {
		if(isFlabby(node)) super.caseAUnaryForRightStat(node);
	}
	public void caseAUnaryTerm(AUnaryTerm node) {
		if(isFlabby(node)) super.caseAUnaryTerm(node);
	}
	public void caseAVarDecInnerDeclaration(AVarDecInnerDeclaration node) {
		if(isFlabby(node)) super.caseAVarDecInnerDeclaration(node);
	}
	public void caseAVoidBasicType(AVoidBasicType node) {
		if(isFlabby(node)) super.caseAVoidBasicType(node);
	}
	public void caseAWhileStatement(AWhileStatement node) {
		if(isFlabby(node)) super.caseAWhileStatement(node);
	}
	public void caseAXorBoolExpression(AXorBoolExpression node) {
		if(isFlabby(node)) super.caseAXorBoolExpression(node);
	}
	public void caseStart(Start node) {
		super.caseStart(node);
	}
}
