/*
 * Created on 21-Dec-2004
 */
package kenya.eclipse.debug;

import java.io.File;

import kenya.eclipse.KenyaPlugin;

import org.eclipse.core.runtime.jobs.Job;

/**
 * @author Thomas Timbul
 */
public abstract class AbstractVKProcess extends Process {
	
	/**
	 * Name given to this process. This may be used to create
	 * workbench jobs or for other identification.
	 */
	public static final String VK_PROCESS_NAME = "VKProcess";
	
	/**
	 * exit value constant denoting unsuccessful termination due to
	 * an unknown, internal error (-101)
	 */
	public static final int INTERNAL_ERROR = -1;
	
	/**
	 * exit value constant denoting normal termination (0)
	 */
	public static final int SUCCESS = 0;
	
	/**
	 * exit value constant denoting normal termination, but an error state
	 * in the underlying program (1)
	 */
	public static final int SUCCESS_BUT_PROGRAM_ERROR = 1;
	
	/**
	 * exit value constant denoting that destroy() was called
	 * before normal termination (102)
	 */
	public static final int PREMATURE_DESTRUCTION = 102;
	
	/**
	 * exit value constant denoting that the process failed
	 * to create necessary pipes and thus failed to execute (103)
	 */
	public static final int STREAM_CREATION_FAILURE = 103;
	
	/**
	 * exit value constant denoting that the working directory
	 * supplied in the constructor was not usable in some way (104)
	 */
	public static final int BAD_WORKING_DIRECTORY = 104;
	
	/**
	 * exit value constant denoting that the code supplied by the
	 * underlying mechanism was unusable for launch (105)
	 */
	public static final int BAD_CODE = 105;
	
	private boolean finished;
	
	protected int returnCode = 0;

	protected File workingDirectory;
	
	protected String className;
	
	protected Job fJob;
	
	/**
	 * constructs a new AbstractVKProcess with given arguments
	 * The working directory must exist and is checked to be a valid directory on disk.
	 * The classname is defaulted to "Temp" if it is null, but must otherwise be a valid
	 * name as defined in kenya.eclipse.KenyaConventions. If this is not the case,
	 * program behaviour is undefined.
	 * @param workDir
	 * @param className
	 */
	public AbstractVKProcess(final String workDir, final String className) {
		this.className = className!=null?className:"Temp";
  	
		workingDirectory = new File(workDir);
		
		if(!(workingDirectory.isDirectory() && workingDirectory.canWrite())) {
			//unusable directory, try to create it
			if(!workingDirectory.mkdirs()) {
				KenyaPlugin.log("Launch of "+this.className+" failed:\n" +
						"Could not access working directory: " + workDir);
				finish(BAD_WORKING_DIRECTORY);
				return;
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Process#exitValue()
	 */
	public int exitValue() {
		if(!hasFinished()) {
			throw new IllegalThreadStateException();
		}
		return returnCode;
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Process#waitFor()
	 */
	public int waitFor() throws InterruptedException {
		while(!hasFinished()) {
			synchronized(this) {
				this.wait();
			}
		}
		return exitValue();
	}
	
	/**
	 * Sets exit value of this process and notifies objects waiting in <code>waitFor()</code>.<br>
	 * Notice that this method also calls <code>dispose()</code> just before it returns<br>
	 * Calling finish mutliple times has no effect and does not overwrite the return
	 * status of the process; ie, if this method is called with an error code AFTER
	 * previously being called with the <code>SUCCESS</code> status, the exitValue received
	 * by objects calling <code>exitValue()</code> is <code>SUCCESS</code>.
	 * 
	 * @param exit the exitValue of this process
	 */
	protected void finish(int exit) {
		if(!finished) {
			synchronized(this) {
				this.returnCode = exit;
				this.finished = true;
				this.notifyAll();
			}
			//cleanup
			
			if(fJob!=null) {
				fJob.cancel();
			}
		}
		dispose();
	}
	
	protected abstract void dispose();
	
//	/**
//	 * cleans up any resources that might have been occupied by this process.
//	 * Implementors should call back to <code>super.dispose()</code> at the end of
//	 * their implementation of this method.
//	 */
//	protected void dispose() {
//		if(fJob!=null) {
//			fJob.sleep();
//			fJob.cancel();
//			fJob = null;
//		}
//	}
	
	/**
	 * returns whether or not this process has terminated
	 * @return true if this process has terminated (exitValue does not throw an Exception)
	 * and false if it hasn't (exitValue throws an Exception)
	 */
	public synchronized boolean hasFinished() {
		return finished;
	}
	
	
	protected abstract IProcessStreamManager getStreamManager();
	
	
}
