/*
 * Created on 13-Dec-2004
 */
package kenya.eclipse.debug;

import java.io.File;
import java.util.ArrayList;

import kenya.eclipse.KenyaConventions;
import mediator.ICheckedCode;
import mediator.IJavaCode;
import mediator.subscription.ICheckedCodeReceiver;
import mediator.subscription.IJavaCodeReceiver;
import mediator.subscription.ISubscriber;
import mediator.subscription.ISubscriptionData;
import mediator.subscription.MediationService;
import mediator.subscription.MediationSubscription;
import mediator.subscription.MediationTerms;
import mediator.subscription.MediatorInput;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.util.Assert;

/**
 * @author Thomas Timbul
 */
public class KenyaRunnerConfiguration {
	
	/**
	 * empty array which signifies no arguments
	 */
	public static final String[] NO_ARGS = new String[0];
	
	/**
	 * the default working directory, &quot;[user.home]/.kenyaTmp&quot;
	 */
	public static final String KENYA_TMP
	  = System.getProperty("user.home") + File.separatorChar + ".kenyaTmp";
	
	/**
	 * the default classname &quot;TempraryClass&quot;
	 */
	public static final String TEMP_CLASS = "TemporaryClass";
	
	private MediatorInput input;
	private String[] args = NO_ARGS;
	private String workingDir = KENYA_TMP;
	private String className = TEMP_CLASS;
	private String projectName = "";
	private boolean stopInMain = false;
	
	/**
	 * instantiates a new Configuration with the given MediatorInput.
	 * All other attributes are initialised to their default values and
	 * should be configured with the provided access methods if necessary.
	 * @param input
	 */
	public KenyaRunnerConfiguration(MediatorInput input) {
		Assert.isLegal(input!=null);
		this.input = input;
	}
	
	/**
	 * sets the program arguments for this configuration
	 * @param args the String array of new program arguments
	 */
	public void setArguments(String[] args) {
		if(args==null) {
			args = NO_ARGS;
		}
		this.args = args;
	}
	
	/**
	 * sets the program arguments for this configuration
	 * This method is equivalent to calling setArguments(args.split("\\s+"));
	 * @param args the new program arguments as a whitespace separated String
	 */
	public void setArguments(String args) {
		setArguments(parseArguments(args));
	}
	
	protected String[] parseArguments(String pargs) {
		if(pargs==null) {
			return NO_ARGS;
		}
		StringBuffer buf = new StringBuffer(pargs.trim());
		ArrayList a = new ArrayList();
		
		while(buf.length()>0) {
			char c = buf.charAt(0);
			if(c=='\"' && buf.indexOf("\"", 1)>0) {
				//get String between quotes (but not single quotes)
				int end = buf.indexOf("\"", 1);
				String arg = buf.substring(1, end);
				buf.delete(0, end+1);
				a.add(arg);
			} else if(Character.isWhitespace(c)){
				//remove whitespace
				buf.deleteCharAt(0);
			} else {
				//get String up to and EX-cluding next whitespace
				StringBuffer arg = new StringBuffer();
				while(buf.length()>0 && !Character.isWhitespace(buf.charAt(0))) {
					arg.append(buf.charAt(0));
					buf.deleteCharAt(0);
				}
				a.add(arg.toString());
			}
		}
		
		return (String[])a.toArray(new String[0]);
	}
	
	/**
	 * returns the program arguments that were set for this configuration
	 * @return the program arguments that were set for this configuration
	 */
	public String[] getArguments() {
		return args;
	}
	
	/**
	 * Returns the IJavaCode associated with this configuration
	 * If the code was not compiled from Kenya, this method will return null.
	 * @return the IJavaCode associated with this configuration or null
	 * if the code is not available
	 */
	public IJavaCode getJavaCode() {
		return receiveCode(input).j;
	}
	
	/**
	 * Returns the ICheckedCode associated with this configuration
	 * If the code was not compiled from Kenya, this method will return null.
	 * @return the ICheckedCode associated with this configuration or null
	 * if the code is not available
	 */
	public ICheckedCode getCheckedCode() {
		return receiveCode(input).c;
	}
	
	/**
	 * receives the CheckedCode and JavaCode into a TempCodeReceiver and
	 * returns it
	 * @param minput MediatorInput for which to retrieve code
	 * @return a TempCodeReceiver containing CheckedCode and JavaCode for the given input
	 */
	protected TempCodeReceiver receiveCode(MediatorInput minput) {
		TempCodeReceiver r = new TempCodeReceiver();
		MediationTerms t = new MediationTerms(minput, r, r);
		MediationSubscription s = new MediationSubscription(new NullSubscriber(), t);
		String id = MediationService.getInstance().addSubscription(s);
		MediationService.getInstance().provideSyncSubscriptionUpdate(id);
		MediationService.getInstance().cancelSubscription(id);
		return r;
	}
	
	/**
	 * ISubscriber which discards any update
	 */
	protected class NullSubscriber implements ISubscriber {
		/* (non-Javadoc)
		 * @see mediator.subscription.ISubscriber#subscriptionUpdate(mediator.subscription.ISubscriptionData)
		 */
		public void subscriptionUpdate(ISubscriptionData data) {
		}
	}
	
	/**
	 * when registered in a MediationSubscription this class can receive both
	 * ICheckedCode and IJavaCode which are then accessible through its protected fields.
	 * These fields are null upon initialisation.
	 * @see kenya.mediator.subscription.MediationSubscription
	 * @see kenya.mediator.subscription.MediationTerms
	 */
	protected class TempCodeReceiver implements IJavaCodeReceiver, ICheckedCodeReceiver {
		protected IJavaCode j = null;
		protected ICheckedCode c = null;
		
		/* (non-Javadoc)
		 * @see mediator.subscription.IJavaCodeReceiver#setJavaCode(mediator.IJavaCode)
		 */
		public synchronized void setJavaCode(IJavaCode code) {
			this.j = code;
		}

		/* (non-Javadoc)
		 * @see mediator.subscription.ICheckedCodeReceiver#setCheckedCode(mediator.ICheckedCode)
		 */
		public void setCheckedCode(ICheckedCode code) {
			this.c = code;
		}
	}
	
	/**
	 * sets the classname associated with this configuration.
	 * The name will be set to the default if the argument is an invalid name
	 * as defined by the KenyaConventions.
	 * @param clazz the new classname to set
	 */
	public void setClassName(String clazz) {
		//check its ok to use
		IStatus st = KenyaConventions.validateKenyaTypeName(clazz);
		if(st.getSeverity()==IStatus.ERROR) {
			clazz = TEMP_CLASS;
		}
		className = clazz;
	}
	
	/**
	 * gets the classname associated with this configuration. This is
	 * the name that is given to the main type of the execution.
	 * @return the classname associated with this configuration
	 */
	public String getClassName() {
		return className;
		
	}
	
	/**
	 * sets the working directory associated with this configuration.
	 * The directory is set to the default if the argument is null or if
	 * it resolves to a plain file on the file system.
	 * @param dir the new working directory to set
	 */
	public void setWorkingDirectory(String dir) {
		if(dir==null) {
			dir = KENYA_TMP;
		} else {
			File f = new File(dir);
			if(f.isFile()) {
				dir = KENYA_TMP;
			}
		}
		workingDir = dir;
	}
	
	/**
	 * returns the working directory associated with this configuration.
	 * @return the working directory associated with this configuration
	 */
	public String getWorkingDirectory() {
		return workingDir;
	}
	
	/**
	 * @param stopInMain whether or not immediately after beginning execution
	 */
	public void setStopInMain(boolean stopInMain) {
		this.stopInMain = stopInMain;
	}
	
	/**
	 * whether or not immediately after beginning execution. The default
	 * value is false and has to be set to true using setStopInMain(boolean)s
	 * @return true if the intent is to stop immediately after beginning execution
	 * and false otherwise
	 */
	public boolean getStopInMain() {
		return stopInMain;
	}
	
	public void setProjectName(String name) {
		projectName = name!=null?name:"";
	}
	
	/**
	 * @return the name of the project which this runner originates from
	 */
	public String getProjectName() {
		return projectName;
	}
	
}
