/*
 * Created on 25-Dec-2004
 */
package kenya.eclipse.debug.model;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;

import kenya.eclipse.debug.AbstractVKProcess;
import kenya.eclipse.debug.AdapterDebugTarget;
import kenya.eclipse.debug.KenyaRemote;
import kenya.eclipse.debug.bridge.IBridgeCommunicator;
import kenya.eclipse.debug.bridge.ICommunicationBridge;
import kenya.eclipse.debug.launcher.IKenyaLaunchConfigurationConstants;
import kenya.eclipse.debug.model.breakpoints.KenyaLineBreakpoint;
import kenya.interpreter.util.InterpreterInspectorListContentProvider;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.stackMachine.IStackMachineInformationProvider;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IMarkerDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IBreakpointManager;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IThread;
import org.wellquite.kenya.stackMachine.StackMachine;

/**
 * @author Thomas Timbul
 */
public class KenyaDebugTarget extends AdapterDebugTarget implements
	                                    IBridgeCommunicator {
	//threads
	private KenyaThread fThread;
	
	private IThread[] fThreads;
	private IThread[] noThreads = new IThread[0]; //used in getThreads()
	
	private ICommunicationBridge fBridge;
	
	private InterpreterInspectorListContentProvider inspector;
	
	private HashMap fBreakpoints;
	
	private boolean disposed = false;
	
	/**
	 * creates a new KenyaDebugTarget with given settings. Note that the
	 * bridge MUST have been setup and contain exactly ONE port which
	 * implements <code>KenyaRemote</code>.
	 * @param launch the ILaunch object
	 * @param process the IProcess object
	 * @param bridge bridge to use to retrieve debug specific data from
	 * @throws CoreException
	 */
	public KenyaDebugTarget(ILaunch launch, AbstractVKProcess process, String name,
			ICommunicationBridge bridge, boolean stop) throws CoreException {
		super(launch, process, name);
		fTarget = this; //workaround
		
		installBreakPoints();
		
		setBridge(bridge);
		
		//get the StackMachine
		StackMachine sm = null;
		IStackMachineInformationProvider ismip = null;
		
		try {
			sm = getStackMachine();
			//stop execution immediately, so we have time to setup
			sm.setStepMode(true);
			ismip = getStackMachineInformationProvider();
		} catch(InvocationTargetException e) {
			//e.printStackTrace();
			abort("Could not access remote StackMachine", e);
		}
		inspector = KenyaDebugModel.getInspectorListContentProvider();
		
		DebugPlugin.getDefault().getBreakpointManager().addBreakpointListener(
				this);
		
		fThread = new KenyaThread(this, inspector, sm, ismip);
		fThreads = new IThread[]{fThread};
		
		if(!stop) { //do the right thing
			fThread.resume();
		} else {
			fireSuspendEvent(DebugEvent.CLIENT_REQUEST);
		}
		
	}
	
	/////////////////////////////////////////////
	/////////////////////////////////////////////
	//
	// methods inherited from IDebugTarget
	//
	/////////////////////////////////////////////
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDebugTarget#getThreads()
	 */
	public IThread[] getThreads() throws DebugException {
		return !disposed?fThreads:noThreads;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDebugTarget#hasThreads()
	 */
	public boolean hasThreads() throws DebugException {
		return disposed;//fThread!=null;
	}
	
	/////////////////////////////////////////////
	/////////////////////////////////////////////
	//
	// methods inherited from IBridgeCommunicator (and related)
	//
	/////////////////////////////////////////////
	
	/*
	 *  (non-Javadoc)
	 * @see kenya.eclipse.debug.bridge.IBridgeCommunicator#setBridge(kenya.eclipse.debug.bridge.ICommunicationBridge)
	 */
	public void setBridge(ICommunicationBridge bridge) {
		fBridge = bridge;
		fBridge.addPort(this);
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.debug.bridge.IBridgeCommunicator#removeBridge(kenya.eclipse.debug.bridge.ICommunicationBridge)
	 */
	public void removeBridge(ICommunicationBridge bridge) {
		if(fBridge==bridge) {
			fBridge = null;
		}
	}
	
	/*
	 *  (non-Javadoc)
	 * @see kenya.eclipse.debug.bridge.IBridgeCommunicator#getBridge()
	 */
	public ICommunicationBridge getBridge() {
		return fBridge;
	}
	
	/*
	 *  (non-Javadoc)
	 * @see kenya.eclipse.debug.bridge.IBridgeCommunicator#invokeRemote(java.lang.String, java.lang.Object[])
	 */
	public Object invokeRemote(String methName, Object[] args)
	throws InvocationTargetException {
		if (methName == null) {
			throw new InvocationTargetException(new NullPointerException(
			"methName is null"));
		} else {
			return null;
		}
	}
	
	/**
	 * invokes the given method with given parameters on the ICommunicationBridge
	 * registered through setBridge.
	 * This is equivalent to directly calling <code>getBridge().invokeRemote(methName, args)</code>
	 * 
	 * @return result of the method call.
	 * @see ICommunicationBridge#invokeRemote(IBridgeCommunicator, String, Object[])
	 */
	protected Object invokeOnBridge(String methName, Object[] args)
	throws InvocationTargetException {
		if (fBridge == null) {
			throw new InvocationTargetException(new NullPointerException(
			"bridge not set"));
		} else {
			return fBridge.invokeRemote(this, methName, args);
		}
	}
	
	public StackMachine getStackMachine() throws InvocationTargetException {
		return (StackMachine)invokeOnBridge(KenyaRemote.GET_STACK_MACHINE, null);
	}
	
	public IStackMachineInformationProvider getStackMachineInformationProvider() throws InvocationTargetException {
		return (IStackMachineInformationProvider)
        invokeOnBridge(KenyaRemote.GET_STACK_MACHINE_INFO_PROVIDER, null);
	}
	
	/////////////////////////////////////////////
	/////////////////////////////////////////////
	//
	// other inherited methods
	//
	/////////////////////////////////////////////
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#canTerminate()
	 */
	public boolean canTerminate() {
		try {
			return ((Boolean)invokeOnBridge(KenyaRemote.CAN_TERMINATE, null)).booleanValue();
		} catch(InvocationTargetException e) {
			return false;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#isTerminated()
	 */
	public boolean isTerminated() {
		boolean term = false;
		try {
			term = ((Boolean)invokeOnBridge(KenyaRemote.IS_TERMINATED, null)).booleanValue();
		} catch(InvocationTargetException e) {
		}
		if(term && !disposed) {
			dispose();
		}
		return term;
	}
	
	protected synchronized void dispose() {
		disposed = true;
		DebugPlugin.getDefault().getBreakpointManager().removeBreakpointListener(this);
		fThread = null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#terminate()
	 */
	public void terminate() throws DebugException {
		try {
			invokeOnBridge(KenyaRemote.TERMINATE, null);
		} catch(InvocationTargetException e) {
			abort("communication failed", e);
		} finally {
			dispose();
			fireTerminateEvent();
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#canResume()
	 */
	public boolean canResume() {
		return !disposed && fThread.canResume();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#canSuspend()
	 */
	public boolean canSuspend() {
		return !disposed && fThread.canSuspend();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#isSuspended()
	 */
	public boolean isSuspended() {
		return !disposed && fThread.isSuspended();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#resume()
	 */
	public void resume() throws DebugException {
		if(fThread!=null) {
			fThread.resume();
		} else {
			abort("target has already terminated", null);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#suspend()
	 */
	public void suspend() throws DebugException {
		if(fThread!=null) {
			fThread.suspend();
		} else {
			abort("target has already terminated", null);
		}
	}
	
	////////////////////////////////////
	////////////////////////////////////
	//
	// breakpoints
	//
	////////////////////////////////////
	
	private void installBreakPoints() {
		
		IBreakpointManager manager = DebugPlugin.getDefault().getBreakpointManager();
		IBreakpoint[] points = manager.getBreakpoints(getModelIdentifier());
		
		fBreakpoints = new HashMap(points.length);
		
		for(int i = 0; i < points.length; i++) {
			IBreakpoint bp = points[i];
			if(supportsBreakpoint(bp)) {
				breakpointAdded(bp);
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDebugTarget#supportsBreakpoint(org.eclipse.debug.core.model.IBreakpoint)
	 */
	public boolean supportsBreakpoint(IBreakpoint breakpoint) {
		if (breakpoint.getModelIdentifier().equals(KenyaDebugModel.getModelIdentifier())) {
			try {
				String program = getLaunch().getLaunchConfiguration().getAttribute(IKenyaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME, (String)null);
				if (program != null) {
					IMarker marker = breakpoint.getMarker();
					if (marker != null) {
						IPath p = new Path(program);
						return marker.getResource().getFullPath().equals(p);
					}
				}
			} catch (CoreException e) {
			}			
		}
		return false;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.IBreakpointListener#breakpointAdded(org.eclipse.debug.core.model.IBreakpoint)
	 */
	public void breakpointAdded(IBreakpoint breakpoint) {
		if(supportsBreakpoint(breakpoint)) {
			KenyaLineBreakpoint lb = (KenyaLineBreakpoint)breakpoint;
			try {
				ISourceCodeLocation loc = lb.getRealLocation();
				fBreakpoints.put(loc, lb);
			} catch(CoreException e) {
				//ignore
			}
			
		}
		
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.IBreakpointListener#breakpointRemoved(org.eclipse.debug.core.model.IBreakpoint, org.eclipse.core.resources.IMarkerDelta)
	 */
	public void breakpointRemoved(IBreakpoint breakpoint, IMarkerDelta delta) {
		if(supportsBreakpoint(breakpoint)) {
			KenyaLineBreakpoint lb = (KenyaLineBreakpoint)breakpoint;
			try {
				ISourceCodeLocation loc = lb.getRealLocation();
				fBreakpoints.remove(loc);
			} catch(CoreException e) {
				//ignore
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.IBreakpointListener#breakpointChanged(org.eclipse.debug.core.model.IBreakpoint, org.eclipse.core.resources.IMarkerDelta)
	 */
	public void breakpointChanged(IBreakpoint breakpoint, IMarkerDelta delta) {
		breakpointAdded(breakpoint);
	}
	
	public boolean breakpointAt(ISourceCodeLocation loc) {
		return fBreakpoints!=null && fBreakpoints.get(loc)!=null;
	}
	
}