/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package kenya.eclipse.multieditor.kenya;

import java.util.Iterator;
import java.util.ResourceBundle;

import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.style.KenyaStyleManager;

import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.IAnnotationAccessExtension;
import org.eclipse.jface.text.source.IVerticalRulerInfo;
import org.eclipse.ui.editors.text.EditorsUI;
import org.eclipse.ui.texteditor.AbstractMarkerAnnotationModel;
import org.eclipse.ui.texteditor.AnnotationPreference;
import org.eclipse.ui.texteditor.AnnotationPreferenceLookup;
import org.eclipse.ui.texteditor.ITextEditor;
import org.eclipse.ui.texteditor.ITextEditorExtension;
import org.eclipse.ui.texteditor.SelectMarkerRulerAction;

/**
 * Action which gets triggered when selecting (annotations) in the vertical ruler.
 * 
 * <p>
 * Was originally called >code>JavaSelectMarkerRulerAction</code>.
 * </p>
 */
public class KenyaSelectAnnotationRulerAction extends SelectMarkerRulerAction {

	private ITextEditor fTextEditor;
	private Position fPosition;
	private AnnotationPreferenceLookup fAnnotationPreferenceLookup;
	private IPreferenceStore fStore;
	private boolean fHasCorrection;
	private ResourceBundle fBundle;
	
	public KenyaSelectAnnotationRulerAction(ResourceBundle bundle, String prefix, ITextEditor editor, IVerticalRulerInfo ruler) {
		super(bundle, prefix, editor, ruler);
		fBundle= bundle;
		fTextEditor= editor;
		
		fAnnotationPreferenceLookup= EditorsUI.getAnnotationPreferenceLookup();
		fStore= KenyaPlugin.getDefault().getCombinedPreferenceStore();
		
		//WorkbenchHelp.setHelp(this, IJavaHelpContextIds.JAVA_SELECT_MARKER_RULER_ACTION);
	}
	
	public void run() {
		if (fHasCorrection) {
			ITextOperationTarget operation= (ITextOperationTarget) fTextEditor.getAdapter(ITextOperationTarget.class);
			final int opCode= KenyaEditor.CORRECTIONASSIST_PROPOSALS;
			if (operation != null && operation.canDoOperation(opCode)) {
				fTextEditor.selectAndReveal(fPosition.getOffset(), fPosition.getLength());
				operation.doOperation(opCode);
			}
			return;
		}

		super.run();
	}
	
	public void update() {
		findJavaAnnotation();
		setEnabled(true); // super.update() might change this later
		
		if (fHasCorrection) {
//			if (fAnnotation instanceof AssistAnnotation)
//				initialize(fBundle, "KenyaSelectAnnotationRulerAction.QuickAssist."); //$NON-NLS-1$
//			else
				initialize(fBundle, "KenyaSelectAnnotationRulerAction.QuickFix."); //$NON-NLS-1$
			return;
		}
		
		initialize(fBundle, "KenyaSelectAnnotationRulerAction.GotoAnnotation."); //$NON-NLS-1$;
		super.update();
	}
	
	private void findJavaAnnotation() {
		fPosition= null;
		fHasCorrection= false;
		
		AbstractMarkerAnnotationModel model= getAnnotationModel();
		IAnnotationAccessExtension annotationAccess= getAnnotationAccessExtension();
		
		IDocument document= getDocument();
		if (model == null)
			return ;
		
		Iterator iter= model.getAnnotationIterator();
		int layer= Integer.MIN_VALUE;
		
		while (iter.hasNext()) {
			Annotation annotation= (Annotation) iter.next();
			if (annotation.isMarkedDeleted())
				continue;
				
			int annotationLayer= annotationAccess.getLayer(annotation);
			if (annotationAccess != null) {
				if (annotationLayer < layer)
					continue;
			}

			Position position= model.getPosition(annotation);
			if (!includesRulerLine(position, document))
				continue;
			
			boolean isReadOnly= fTextEditor instanceof ITextEditorExtension
			               && ((ITextEditorExtension)fTextEditor).isEditorInputReadOnly();
			if (!isReadOnly
					&& (
//						((hasAssistLightbulb && annotation instanceof AssistAnnotation)
//						||
						KenyaStyleManager.hasCorrections(annotation)))
				//)
				{
				fPosition= position;
				fHasCorrection= true;
				layer= annotationLayer;
				continue;
			} else {
				AnnotationPreference preference= fAnnotationPreferenceLookup.getAnnotationPreference(annotation);
				if (preference == null)
					continue;
				
				String key= preference.getVerticalRulerPreferenceKey();
				if (key == null)
					continue;
				
				if (fStore.getBoolean(key)) {
					fPosition= position;
					fHasCorrection= false;
					layer= annotationLayer;
				}
			}
		}
	}
}

