package kenya.eclipse.style;

import java.util.HashMap;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import kenya.eclipse.KenyaPlugin;
import kenya.eclipse.style.checkerimpl.IStyleChecker;
import kenya.eclipse.style.checkerimpl.StyleCheckerDescriptor;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.ui.texteditor.SimpleMarkerAnnotation;

/**
 * main Kenya Style Guide plugin class
 */
public class KenyaStyleManager {
	
	public static final String KENYA_STYLE_ID = "kenya.eclipse.KenyaStyleManager";
	
	//StyleCheckRegistry that holds registered style checkers
	private static StyleCheckRegistry fStyleRegistry;
	
	//ResolutionManager to keep track of IMarkerResolutions
	private static KenyaStyleResolutionManager fResolutionManager;
	
	/**
	 * The constructor.
	 */
	public KenyaStyleManager() {
		super();
	}
	
	public static StyleCheckRegistry getStyleCheckRegistry() {
		if(fStyleRegistry==null) {
			initialiseStyleRegistry();
		}
		return fStyleRegistry;
	}
	
	private static void initialiseStyleRegistry() {
		fStyleRegistry = new StyleCheckRegistry();
		IExtensionPoint extensionPoint
		  = Platform.getExtensionRegistry().getExtensionPoint(KenyaPlugin.getPluginId(),
		  		KenyaStyleConstants.EXTENSION_POINT_STYLE_CHECKER);
		IConfigurationElement[] infos = extensionPoint.getConfigurationElements();
		for (int i = 0; i < infos.length; i++) {
			try {
				String id = infos[i].getAttribute("id");
				String name = infos[i].getAttribute("name");
				String defEnabled = infos[i].getAttribute("defaultEnabled");
				String namespace = infos[i].getDeclaringExtension().getNamespace();
				
				IStyleChecker checker = (IStyleChecker)infos[i].createExecutableExtension("class");
				
				//build the custom attributes map
				IConfigurationElement[] custom = infos[i].getChildren("customAttribute");
				HashMap customAttributes = new HashMap(custom.length);
				for(int j = 0; j < custom.length; j++) {
					IConfigurationElement data = custom[j];
					String cname  = data.getAttribute("name");
					String cvalue = data.getAttribute("value");
					customAttributes.put(cname, cvalue);
				}
				//configure the style checker
				checker.configure(customAttributes);
				
				QualifiedName qn = new QualifiedName(namespace, id);
				
				boolean enabled
				  = defEnabled==null || defEnabled.equals(KenyaStyleConstants.ENABLED);
				
				StyleCheckerDescriptor desc
				  = new StyleCheckerDescriptor(qn, name, checker, enabled);
				fStyleRegistry.registerStyleChecker(desc);
			} catch(CoreException e) {
				KenyaPlugin.log(e);
			} catch(ClassCastException e) {
				KenyaPlugin.log(e);
			}
		}
	}
	
	/**
	 * Returns the string from the plugin's resource bundle,
	 * or 'key' if not found.
	 */
	public static String getResourceString(String key) {
		ResourceBundle bundle = KenyaPlugin.getDefault().getResourceBundle();
		try {
			return (bundle != null) ? bundle.getString(key) : key;
		} catch (MissingResourceException e) {
			return key;
		}
	}
	
	public static KenyaStyleResolutionManager getResolutionManager() {
		if (fResolutionManager == null) {
			fResolutionManager = new KenyaStyleResolutionManager();
		}
		
		return fResolutionManager;
	}

	/**
	 * @param annotation
	 * @return
	 */
	public static boolean hasCorrections(Annotation annotation) {
		if (annotation instanceof SimpleMarkerAnnotation) {
			SimpleMarkerAnnotation a = (SimpleMarkerAnnotation)annotation;
			return getResolutionManager().hasResolutions(a.getMarker());
		}
		return false;
	}
	
}
