/*
 * Created on 23-Feb-2005
 */
package kenya.eclipse.style.checks.bool;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import kenya.eclipse.ast.AdvancedPositionFinder;
import kenya.eclipse.ast.NodeToString;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentModificationOperation;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentTextOperation;
import kenya.eclipse.style.StyleWarningResolution;
import kenya.eclipse.style.checkerimpl.AbstractStyleChecker;
import kenya.eclipse.style.checks.StatementSearch;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.ICheckedCode;
import minijava.node.ABlockElseFollow;
import minijava.node.AElsePart;
import minijava.node.AIfStat;
import minijava.node.Node;
import minijava.node.PElseFollow;
import minijava.node.PUnaryExp;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;

/**
 * @author Thomas Timbul
 */
public class EmptyIfElseChecker extends AbstractStyleChecker {
	
	public static final String IF_MESSAGE   = "The 'if' part of this if statement is empty.";
	public static final String ELSE_MESSAGE = "The 'else' part of this if statement is empty.";

	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#configure(java.util.Map)
	 */
	public void configure(Map customAttributes) {
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#performCheck(mediator.ICheckedCode, org.eclipse.core.resources.IFile)
	 */
	public void performCheck(ICheckedCode code, final IFile file) {
		
		IFunction[] functions = code.getFunctions();
		
		ArrayList ifs = new ArrayList();
		
		for(int i = 0; i < functions.length; i++) {
			Node n = (Node)functions[i].getDeclarationNode().clone();
			ifs.addAll(IfSearch.getIfStats(n, false));
		}
		
		if(fResolutionMap==null) {
			fResolutionMap = new HashMap(ifs.size()/2); //about half may have bad style
		}
		
		for(Iterator ifs_iter = ifs.iterator(); ifs_iter.hasNext();) {
			final AIfStat stat = (AIfStat)ifs_iter.next();
			if(stat==null) continue;
			final AElsePart els = (AElsePart)stat.getElsePart();
			if(els==null) continue;
			final PElseFollow f = els.getElseFollow();
			
			if(f==null || !(f instanceof ABlockElseFollow)) {
				//TODO: handle 'else ifs'
				continue;
			}
			
			List statements_if = StatementSearch.getAllStatements(stat.getBlock1());
			
			if(statements_if.size() == 0) { //empty if block
				
				final IDocument doc = getDocument(file);
				
				//the runnable creates the marker and the resolution
				IWorkspaceRunnable runnable = new IWorkspaceRunnable() {
					public void run(IProgressMonitor monitor) throws CoreException {
						ISourceCodeLocation pp = AdvancedPositionFinder.getFullLocation(stat.getIf(), doc);
						IMarker marker;
						try {
							marker = createKenyaStyleMarker(
									file,
									pp,
									IF_MESSAGE
							);
						} catch(CoreException e) {
							return;
						}
						
						DocumentModificationOperation o = new DocumentModificationOperation();
						
						ISourceCodeLocation boolLocation
						  = AdvancedPositionFinder.getFullLocation(stat.getBoolExpression(), doc);
						Position boolPos = calculatePosition(boolLocation, doc);
						Node exp = BoolNodeTools.getLargestBoolNode(stat.getBoolExpression());
						PUnaryExp negation = (PUnaryExp)BoolNodeTools.getParent(exp, PUnaryExp.class);
						negation = BoolNodeTools.negate(negation);
						String neg = NodeToString.toString(negation);
						o.addOperation(
								DocumentTextOperation.newTextReplacement(boolPos.offset, boolPos.length, neg)
						);
						
						ISourceCodeLocation block1loc
						  = AdvancedPositionFinder.getFullLocation(stat.getBlock1(), doc);
						Position block1pos = calculatePosition(block1loc, doc);
						
						ABlockElseFollow follow = (ABlockElseFollow)f;
						
						ISourceCodeLocation block2loc
					    = AdvancedPositionFinder.getFullLocation(follow, doc);
						Position block2pos = calculatePosition(block2loc, doc);
						
						int start = block1pos.offset;
						int length = block2pos.offset - start; 
						
						o.addOperation(
								DocumentTextOperation.newTextDeletion(start, length)
							);
						
						StyleWarningResolution res = new StyleWarningResolution("replace with else block contents and negate condition", marker, o) {
							public String getDescription() {
								return "<p><b>Explanation:</b></p>" + 
								"<p>The 'if' part of this if statement is empty. Although this " +
								"can be optimised by a compiler, the code would be more readable " +
								"if the empty block was removed.</p>";
							}
						};
						
						fResolutionMap.put(marker, res);
					}
				};
				
				runMarkerUpdate(runnable);
			}
		}
	}
}
