/*
 * Created on 22-Jan-2005
 */
package kenya.eclipse.style.checks.bool;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import kenya.eclipse.ast.AdvancedPositionFinder;
import kenya.eclipse.ast.NodeToString;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentModificationOperation;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentTextOperation;
import kenya.eclipse.style.StyleWarningResolution;
import kenya.eclipse.style.checkerimpl.AbstractStyleChecker;
import kenya.eclipse.style.checks.StatementSearch;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.ICheckedCode;
import minijava.node.AElsePart;
import minijava.node.AIfStat;
import minijava.node.AReturnStatement;
import minijava.node.Node;
import minijava.node.PElseFollow;
import minijava.node.PExpression;
import minijava.node.PUnaryExp;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;

/**
 * @author Thomas Timbul
 */
public class IfReturnElseReturnChecker extends AbstractStyleChecker {
	
	public static final String MESSAGE = "This 'if' statement is unnecessary and can be replaced with a single return statement";

	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#configure(java.util.Map)
	 */
	public void configure(Map customAttributes) {
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#performCheck(mediator.ICheckedCode, org.eclipse.core.resources.IFile)
	 */
	public void performCheck(ICheckedCode code, final IFile file) {
		
		IFunction[] functions = code.getFunctions();
		
		ArrayList ifs = new ArrayList();
		
		for(int i = 0; i < functions.length; i++) {
			Node n = (Node)functions[i].getDeclarationNode().clone();
			ifs.addAll(IfSearch.getIfStats(n, false));
		}
		
		if(fResolutionMap==null) {
			fResolutionMap = new HashMap(ifs.size()/2); //about half may have bad style
		}
		
		for(Iterator ifs_iter = ifs.iterator(); ifs_iter.hasNext();) {
			final AIfStat stat = (AIfStat)ifs_iter.next();
			
			AReturnStatement block1Returns;
			
			List statements = StatementSearch.getAllStatements(stat.getBlock1());
			
			if(statements.size() <= 1) {
				block1Returns = (statements.size()==1
						          && statements.get(0) instanceof AReturnStatement)
									?(AReturnStatement)statements.get(0)
									:null;
				if(block1Returns==null) {
					continue;
				}
				
				AReturnStatement block2Returns;
				
				AElsePart els = (AElsePart)stat.getElsePart();
				PElseFollow f = els.getElseFollow();
				
				statements = StatementSearch.getAllStatements(f);
				
				if(statements.size() <= 1) {
					block2Returns = (statements.size()==1
							          && statements.get(0) instanceof AReturnStatement)
										?(AReturnStatement)statements.get(0)
										:null;
					if(block2Returns==null) {
						continue;
					}
					
					final PExpression r1e = block1Returns.getExpression();
					final PExpression r2e = block2Returns.getExpression();
					
					final IDocument doc = getDocument(file);
					ISourceCodeLocation loc = AdvancedPositionFinder.getFullLocation(stat, doc);
					final Position pos = calculatePosition(loc, doc);
					
					//the runnable creates the marker and the resolution
					IWorkspaceRunnable runnable = new IWorkspaceRunnable() {
						public void run(IProgressMonitor monitor) throws CoreException {
							ISourceCodeLocation pp = AdvancedPositionFinder.getFullLocation(stat.getIf(), doc);
							IMarker marker;
							try {
								marker = createKenyaStyleMarker(
										file,
										pp,
										MESSAGE
									);
							} catch(CoreException e) {
								return;
							}
							
							String repl = null;
							
							if(BoolNodeTools.isTrue(r1e) && BoolNodeTools.isFalse(r2e)) {
								//return the if condition
								repl = NodeToString.toString(stat.getBoolExpression());
								
							} else if(BoolNodeTools.isTrue(r2e) && BoolNodeTools.isFalse(r1e)) {
								//return NOT the if condition
								
								Node n = BoolNodeTools.getLargestBoolNode(stat.getBoolExpression());
								PUnaryExp unary = (PUnaryExp)BoolNodeTools.getParent(n, PUnaryExp.class);
								repl = NodeToString.toString(BoolNodeTools.negate(unary));
								
							} else if(BoolNodeTools.limitedAreEqual(r1e, r2e)) {
								repl = NodeToString.toString(r1e);
							} else {
								marker.delete();
								return;
							}
							
							repl = MessageFormat.format("return {0};", new String[]{repl});
							//System.out.println("can replace if statement with '" + repl+"'");
							DocumentTextOperation op = DocumentTextOperation.newTextReplacement(pos.offset, pos.length, repl);
							
							DocumentModificationOperation o = new DocumentModificationOperation();
							o.addOperation(op);
							
							StyleWarningResolution res = new StyleWarningResolution("replace with "+repl, marker, o) {
								public String getDescription() {
									return "<p><b>Explanation:</b></p>" +
											"<p>You are testing whether a boolean expression is true or false and then return " +
											"true or false accordingly. However, since the boolean expression itself already holds " +
											"exactly that truth value, it is equivalent to simply return this expression.<br>" +
											"Not only would this make the code look nicer and more concise, but it also loses you less marks for " +
											"using 'bad style'! :p</p>";
								}
							};
							
							fResolutionMap.put(marker, res);
						}
					};
					
					runMarkerUpdate(runnable);
				}
			}
		}
		
		
		
	}
}
