/*
 * Created on 19-Mar-2005
 */
package kenya.eclipse.style.checks.scope;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import kenya.eclipse.KenyaConventions;
import kenya.eclipse.ast.NodeTools;
import kenya.eclipse.ast.OccurrencesFinder;
import kenya.eclipse.ast.SimpleASTProvider;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentModificationOperation;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentTextOperation;
import kenya.eclipse.multieditor.kenya.util.LocationUtils;
import kenya.eclipse.style.StyleWarningResolution;
import kenya.eclipse.style.checkerimpl.AbstractStyleChecker;
import kenya.sourceCodeInformation.interfaces.IClass;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.sourceCodeInformation.interfaces.IVariable;
import kenya.sourceCodeInformation.util.SourceCodeLocation;
import mediator.ICheckedCode;
import minijava.analysis.DepthFirstAdapter;
import minijava.node.AArrayDecInnerDeclaration;
import minijava.node.AVarDecInnerDeclaration;
import minijava.node.Start;
import minijava.node.TIdentifier;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;

/**
 * @author Thomas Timbul
 */
public class ShadowedConstantsChecker extends AbstractStyleChecker {
	
	public static final String MESSAGE =
		"<p><b>Explanation:</b></p>" +
		"<p>You have declared a local variable with the same name as a constant.<br>" +
		"This is called &quot;Shadowing&quot; and means that the constant will not" +
		" be accessible inside this method, since the <i>local</i> scope has precedence" +
		" over the <i>global</i> one.<br>" +
		"Although it is possible in Java to still reference the constant, it is" +
		" generally advisable to avoid this situation altogether. The easiest way is" +
		" to simply rename the local variable.<br>" +
		"This fix will ask you for a new name and perform the rename for you." +
		"</p>";
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#configure(java.util.Map)
	 */
	public void configure(Map customAttributes) {
	}

	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#performCheck(mediator.ICheckedCode, org.eclipse.core.resources.IFile)
	 */
	public void performCheck(final ICheckedCode code, final IFile file) {
		
		IVariable[] constants = code.getConstants();
		
		if(constants.length==0) return; //there is nothing to check
		
		String[] constantNames = new String[constants.length];
		for(int i = 0; i < constants.length; i++) {
			constantNames[i] = constants[i].getName();
		}
		
		//sort so we can find stuff faster using binary search
		Arrays.sort(constantNames);
		
		//check variable declarations in all methods to see if they have the
		// same name as any of the constants
		
		final ArrayList offendingLocations // <ISourceCodeLocation>
		  = new ArrayList(constantNames.length);
		
		IFunction[] funs = code.getFunctions();
		
		for(int i = 0; i < funs.length; i++) {
			IFunction fun = funs[i];
			
			IVariable[] params = fun.getArguments();
			for(int j = 0; j < params.length; j++) {
				IVariable param = params[j];
				if(Arrays.binarySearch(constantNames, param.getName())>=0) {
					//the name was found in the constants
					ISourceCodeLocation loc = param.getPosition();
					//this location needs to be shifted by the length of the type
					IClass c = param.getType();
					SourceCodeLocation newloc = new SourceCodeLocation(
							loc.getLineNumber(),
							loc.getColumnNumber()+c.getName().length()+1,
							param.getName().length()
						);
					offendingLocations.add(newloc);
				}
			}
			
			VariableDeclarationSearch vds = new VariableDeclarationSearch();
			
			fun.getDeclarationNode().getBlock().apply(vds);
			
			List arrays = vds.getArrays();
			List vars = vds.getVars();
			
			//search array declarations
			for(Iterator it = arrays.iterator(); it.hasNext();) {
				AArrayDecInnerDeclaration a = (AArrayDecInnerDeclaration)it.next();
				TIdentifier id = a.getIdentifier();
				String name = id.getText();
				if(Arrays.binarySearch(constantNames, name)>=0) {
					//the name was found in the constants
					offendingLocations.add(NodeTools.getLocation(id));
				}
			}
			
			//search variable declarations
			for(Iterator it = vars.iterator(); it.hasNext();) {
				AVarDecInnerDeclaration a = (AVarDecInnerDeclaration)it.next();
				TIdentifier id = a.getIdentifier();
				String name = id.getText();
				if(Arrays.binarySearch(constantNames, name)>=0) {
					//the name was found in the constants
					offendingLocations.add(NodeTools.getLocation(id));
				}
			}
			
		}
		
		
		//ok, now we have a list of 'offending' locations
		
		IWorkspaceRunnable run = new IWorkspaceRunnable() {
			public void run(IProgressMonitor monitor) throws CoreException {
				IDocument doc = getDocument(file);
				
				fResolutionMap = new HashMap(offendingLocations.size());
				
				HashMap inputs = new HashMap(1);
				String key = "Please enter new variable name.";
				String var = "newName";
				inputs.put(key, var);
				HashMap valid = new HashMap(1);
				valid.put(var, new Validator());
				
				for(Iterator it = offendingLocations.iterator(); it.hasNext();) {
					ISourceCodeLocation loc = (ISourceCodeLocation)it.next();
					
					IMarker m = createKenyaStyleMarker(file, loc, "This local variable is 'shadowing' a constant.");
					
					DocumentModificationOperation dmo
					  = new DocumentModificationOperation(inputs, valid);
					
					Start ast = SimpleASTProvider.getAST(code);
					
					OccurrencesFinder of = new OccurrencesFinder();
					of.initialize(ast, code, doc, loc);
					
					List occs = of.perform();
					
					for(Iterator iter = occs.iterator(); iter.hasNext();) {
						TIdentifier tid = (TIdentifier)iter.next();
						ISourceCodeLocation location = NodeTools.getLocation(tid);
						Position pos = LocationUtils.convert(location, doc);
						
						DocumentTextOperation op
						  = DocumentTextOperation.newTextReplacement(pos.offset, pos.length, "${"+var+"}");
						
						dmo.addOperation(op);
					}
					
					StyleWarningResolution res = new StyleWarningResolution("rename local variable", m, dmo) {
						/* (non-Javadoc)
						 * @see kenya.eclipse.style.StyleWarningResolution#getDescription()
						 */
						public String getDescription() {
							return MESSAGE;
						}
					};
					
					fResolutionMap.put(m, res);
					
				}
			}
		};
		
		super.runMarkerUpdate(run);
		
		
	}
	
}

class Validator implements IInputValidator {
	
	public Validator() {}
	
	public String isValid(String txt) {
		if(KenyaConventions.clashesWithReservedWord(txt)) {
			return "Cannot give the same name as a reserved keyword.";
		} else {
			if(txt.length()==0) return null;
			char[] cs = txt.toCharArray();
			boolean valid = Character.isJavaIdentifierStart(cs[0]);
			for(int i = 1; i < cs.length; i++) {
				if(!valid) {
					return "Identifiers must start with an underscore or letter and cannot contain spaces or other special characters.";
				}
				valid = Character.isJavaIdentifierPart(cs[i]);
			}
			return null;
		}
	}
}

class VariableDeclarationSearch  extends DepthFirstAdapter {
	
	private ArrayList arrays = new ArrayList();
	private ArrayList vars = new ArrayList();
	
	public List getArrays() {
		return arrays;
	}
	
	public List getVars() {
		return vars;
	}
	
	/* (non-Javadoc)
	 * @see minijava.analysis.DepthFirstAdapter#caseAArrayDecInnerDeclaration(minijava.node.AArrayDecInnerDeclaration)
	 */
	public void caseAArrayDecInnerDeclaration(AArrayDecInnerDeclaration node) {
		arrays.add(node);
	}
	
	/* (non-Javadoc)
	 * @see minijava.analysis.DepthFirstAdapter#caseAVarDecInnerDeclaration(minijava.node.AVarDecInnerDeclaration)
	 */
	public void caseAVarDecInnerDeclaration(AVarDecInnerDeclaration node) {
		vars.add(node);
	}
	
}
