/*
 * Created on 21-Jan-2005
 */
package kenya.eclipse.style.checks.swit;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import kenya.eclipse.KenyaConstants;
import kenya.eclipse.multieditor.kenya.refactoring.DocumentTextOperation;
import kenya.eclipse.style.StyleWarningResolution;
import kenya.eclipse.style.checkerimpl.AbstractStyleChecker;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.util.SourceCodeLocation;
import mediator.ICheckedCode;
import minijava.node.ADefaultPossibleCase;
import minijava.node.ASwitchBlock;
import minijava.node.ASwitchStatement;
import minijava.node.Node;
import minijava.node.PPossibleCase;
import minijava.node.TSwitch;
import minijava.node.Token;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;

/**
 * @author Thomas Timbul
 */
public class DefaultOmissionChecker extends AbstractStyleChecker {
	
	private static final String MESSAGE = "This switch statement does not have a default case. Consider what happens if none of your other cases match...";
	
	public DefaultOmissionChecker() {
		super();
	}

	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#configure(java.util.Map)
	 */
	public void configure(Map customAttributes) {
	}
	
	/* (non-Javadoc)
	 * @see kenya.eclipse.style.checkerimpl.IStyleChecker#performCheck(mediator.ICheckedCode, org.eclipse.core.resources.IFile)
	 */
	public void performCheck(ICheckedCode code, final IFile file) {
		
		IFunction[] functions = code.getFunctions();
		
		final List bigList = new ArrayList(1); //1 since there won't be many
		
		for(int i = 0; i < functions.length; i++) {
			Node node = (Node)functions[i].getDeclarationNode().clone();
			bigList.addAll(SwitchSearch.getSwitchStatements(node));
		}
		
		statements:
		for(Iterator it = bigList.iterator(); it.hasNext();) {
			ASwitchStatement s = (ASwitchStatement)it.next();
			
			ASwitchBlock block = (ASwitchBlock)s.getSwitchBlock();
			LinkedList cases = block.getPossibleCase();
			
			for(Iterator iter = cases.listIterator(); iter.hasNext();) {
				PPossibleCase element = (PPossibleCase)iter.next();
				
				if(element instanceof ADefaultPossibleCase) {
					//if there is a default clause, then we can skip to the next
					it.remove(); // and remove the current statement
					continue statements;
				}
			}
		}
		
		//bigList now contains only switches without default clause
		
		IWorkspaceRunnable runnable = new IWorkspaceRunnable() {
			public void run(IProgressMonitor monitor) throws CoreException {
				
				for(Iterator it = bigList.iterator(); it.hasNext();) {
					ASwitchStatement s = (ASwitchStatement)it.next();
					
					TSwitch sw = s.getSwitch();
					SourceCodeLocation loc
					  = new SourceCodeLocation(sw.getLine(), sw.getPos(), sw.getText().length());
					
					IMarker m = createKenyaStyleMarker(file, loc, MESSAGE);
					
					Token brace = ( (ASwitchBlock)s.getSwitchBlock() ).getRBrace();
					int line = brace.getLine()-1; //reduce because of indexing
					
					int fixOffset = -1;
					try {
						fixOffset = getDocument(file).getLineOffset(line)+brace.getPos()-1;
					} catch(BadLocationException e) {
						return;
					}
					m.setAttribute(FIX_OFFSET, fixOffset);
					
					DefaultOmissionMarkerResolution r0
					  = new DefaultOmissionMarkerResolution(m);
					
					if(fResolutionMap==null) {
						fResolutionMap = new HashMap(bigList.size());
					}
					
					fResolutionMap.put(m, new StyleWarningResolution[]{r0} );
				}
				
			}
		};
		
		runMarkerUpdate(runnable);
		
	}
	
	class DefaultOmissionMarkerResolution extends StyleWarningResolution {
		
		public DefaultOmissionMarkerResolution(IMarker marker) {
			super("Create default case", marker);
			init(marker);
		}
		
		/* (non-Javadoc)
		 * @see kenya.eclipse.style.resolution.StyleWarningResolution#getDescription()
		 */
		public String getDescription() {
			return "<p><b>Explanation:</b></p>" +
					"<p>You have missed out the <b>default case</b> in a switch. It is duly possible that " +
					"during the execution of your code, none of the other cases match. That event " +
					"is handled by the 'default' case." +
					"<br>If it does not exist, the switch will simply return, the program " +
					"continues executing and you may have no idea what has happened.<br>" +
					"That's why it is a good idea to always have a default case, so that " +
					"you can inform the user (or yourself) that no other case was satisfied.</p>" +
					"<p>Applying this fix will create a default case in the switch block.</p>";
		}
		
		private void init(IMarker marker) {
			IDocument doc = getDocument((IFile)marker.getResource());
			if(doc==null) {
				return;
			}
			
			//find the place where we are supposed to insert our generation
			int offset = marker.getAttribute(FIX_OFFSET, -1);
			
			if(offset<0) {
				return; //no such place defined, then we can't continue
			}
			
			String prefix; //prefix (e.g. indent)
			String postfix; //a postfix (e.g. line delimiter)
			
			String indentUnit = "";
			StringBuffer buf = new StringBuffer(); //what we insert into the document in the end
			
			try {
				int line = doc.getLineOfOffset(offset); //line number
				String delim = doc.getLineDelimiter(line); //line delimiter
				int lineOffset = doc.getLineOffset(line); //line offset
				int length = offset - lineOffset;
				
				//the line up to our insert position
				String lineContent = doc.get(lineOffset, length);
				
				//find the indent of the given line
				String indent = lineContent;
				
				if(indent.charAt(0)=='\t') {
					indentUnit = "\t";
				} else {
					//add tabwidth many spaces
					indentUnit = "";
					for(int i =0; i<KenyaConstants.EDITOR_TAB_WIDTH; i++) {
						indentUnit += ' ';
					}
				}
				
				if("".equals(lineContent.trim())) {
					//then the indent is that of the previous line!
					// (which will be ours +1 unit) - so prefix one unit
					prefix = indentUnit;
					postfix = delim + indent;
				} else {
					int nonWhiteLength = lineContent.trim().length();
					int whiteEnd = lineContent.length() - nonWhiteLength;
					indent = indent.substring(0, whiteEnd);
					prefix = delim + indent;
					
					postfix = delim + indent;
					postfix = postfix.substring(0, postfix.length()-indentUnit.length());
				}
				
				buf.append(prefix).append("default: {").append(delim);
				buf.append(indent).append(indentUnit).append(indentUnit).append("//TODO: fill default case").append(delim);
				buf.append(indent).append(indentUnit).append(indentUnit).append("break;").append(delim);
				buf.append(indent).append(indentUnit).append("}").append(postfix);
				
			} catch(BadLocationException e) {
				return;
			}
			
			fOperation.addOperation(DocumentTextOperation.newTextInsertion(offset, buf.toString()));
		}
		
	}

}
