/*
 * Created on 28-Oct-2004
 */
package kenya.sole.ui.editor;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;

import kenya.sole.ui.core.KenyaUI;
import kenya.sole.ui.editor.util.DefaultHighlightingManager;
import kenya.sole.ui.event.EditorControlListener;
import kenya.sole.ui.event.IModifiableObject;
import kenya.sole.ui.event.IModificationListener;
import kenya.sole.ui.event.ModificationEvent;
import kenya.sole.ui.event.NumbersSelectionListener;
import kenya.sole.ui.util.IHighlightingManager;
import kenya.ui.IButtonManager;

import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.LineStyleListener;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;

/**
 * @author Thomas Timbul
 */
public class EditorPart extends SashForm
												implements ActionListener, IModifiableObject
		{
	
	//****************************
	// do NOT attempt to initialise the variables here, it doesn't
	//   work because of the weird constructor/init structure which
	//   causes variables instantiated in init to be overwritten
	//   with null from the global declaration!!
	//****************************
	
	protected Object lock;
	
	protected LineStyleListener lineStyler;
	protected volatile StyledText lineNumbers;
	protected int lineNumberCount = -1;
	protected StringBuffer lineNumberBuffer;
	
	
	protected int caretOffset = 0;
	
	
	protected IHighlightingManager highlightingManager ;
	
	
	protected IStatusLineManager statusLine;
	protected IButtonManager buttonManager;
	
	
	protected volatile String editorWidgetText = "";
	protected volatile StyledText editorText;
	
	
	protected boolean modified;
	
	
	private boolean isVisible;
	
	
	private final Runnable runIsVisible = new Runnable() {

		public void run() {
			isVisible();
		}
	};
	private final Runnable runGetEditorText = new Runnable() {
		public void run() {
			getEditorText();
		}
	};
	private final Runnable runGetCaretOffset = new Runnable() {
		public void run() {
			getCaretOffset();
		}
	};
	
	private ArrayList modListeners = new ArrayList();
	
	private EditorPart(Composite parent, int style) {
		super(parent, style);
		lock = new Object();
		lineStyler = null;
		lineNumbers = null;
		lineNumberBuffer = new StringBuffer();
	}
	public EditorPart(
				Composite parent,
				int style,
				IStatusLineManager statusLine,
				IButtonManager buttonManager
			) {
		this(parent, style);
		this.statusLine = statusLine;
		this.buttonManager = buttonManager;
		//init();
	}
	
	public void safeSetStatus(final String message) {
		if(statusLine!=null) {
			if (Thread.currentThread()
					.equals(this.getDisplay().getThread())) {
				synchronized (this) {
					statusLine.setMessage(message);
				}
			} else {
				if (!isVisible())
					return;
				this.getDisplay().syncExec(new Runnable() {
	
					public void run() {
						safeSetStatus(message);
					}
				});
			}
		}
	}
	
	public void setLineStyler(LineStyleListener styler) {
		if(editorText!=null) {
			if(lineStyler!=null) {
				editorText.removeLineStyleListener(lineStyler);
			}
			this.lineStyler = styler;
			editorText.addLineStyleListener(lineStyler);
		}
		
	}
	public LineStyleListener getLineStyler() {
		return this.lineStyler;
	}
	
	public void init() {
		this.setLayout(new FillLayout());
		
		final SashForm editorSash = new SashForm(this,
				SWT.HORIZONTAL);
		
		buildLineNumbers(editorSash);
		buildEditorContent(editorSash);
		
		setModified(false);
		
		editorSash.setWeights(new int[] { 1, 18 });
		
		this.setWeights(new int[] { 1 });
		
		editorText.setFocus();
		
		editorSash.addControlListener(new EditorControlListener(editorSash,
				lineNumbers, editorText));
	}
	
	public void safeRedraw() {
		if (Thread.currentThread()
				.equals(this.getDisplay().getThread())) {
			synchronized (this) {
				this.redraw();
			}
		} else {
			if (!isVisible())
				return;
			this.getDisplay().syncExec(
						new Runnable() {
							public void run() {
								safeRedraw();
							}
						}
					);
		}
	}
	
	protected void buildLineNumbers(SashForm parent) {
		lineNumbers = new StyledText(parent, SWT.NO_FOCUS | SWT.READ_ONLY);
		lineNumbers.setEditable(false);
		lineNumbers.setDoubleClickEnabled(false);
		lineNumbers.setEnabled(false);
		lineNumbers.addSelectionListener(new NumbersSelectionListener(lineNumbers));
		lineNumbers.setBackground(DefaultHighlightingManager.WHITE);
	}
	public void rebuildLineNumbers() {
		if (Thread.currentThread()
				.equals(this.getDisplay().getThread())) {
			synchronized (this) {
				if (lineNumberCount != editorText.getLineCount()) {
					int topPixel = lineNumbers.getTopPixel();
					lineNumberBuffer.setLength(0);
					for (int idx = 1; idx <= editorText.getLineCount(); idx++) {
						lineNumberBuffer.append(idx);
						lineNumberBuffer.append(SWT.CR);
					}
					lineNumbers.setText(lineNumberBuffer.toString());
					lineNumbers.setTopPixel(topPixel);
					lineNumberCount = editorText.getLineCount();
				}
			}
		} else {
			if (isVisible()) {
				this.getDisplay().syncExec(
						new Runnable() {
							public void run() {
								rebuildLineNumbers();
							}
						}
					);
			}
		}
	}
	public synchronized void positionLineNumbers() {
		if (lineNumbers.getTopPixel() != editorText.getTopPixel()) {
			lineNumbers.setTopPixel(editorText.getTopPixel());
			lineNumbers.redraw();
		}
	}
	
	protected void buildEditorContent(SashForm parent) {
		editorText = new StyledText(parent, SWT.NULL | SWT.V_SCROLL
				| SWT.H_SCROLL);
	}
	
	public synchronized StyledText getLineNumberWidget() {
		return lineNumbers;
	}
	
	private String getEditorWidgetText() {
		synchronized (lock) {
			return editorWidgetText;
		}
	}
	private void setEditorWidgetText(String editorContentText) {
		synchronized (lock) {
			this.editorWidgetText = editorContentText;
		}
	}
	
	public synchronized StyledText getEditorTextWidget() {
		return editorText;
	}
	public String getEditorText() {
		if (!isVisible())
			return "";
		if (Thread.currentThread().equals(editorText.getDisplay().getThread())) {
			synchronized (this) {
				setEditorWidgetText(editorText.getText());
			}
		} else {
			editorText.getDisplay().syncExec(runGetEditorText);
		}
		return getEditorWidgetText();
	}
	
	private void setCaretOffsetInternal(int offset) {
		synchronized (lock) {
			caretOffset = offset;
		}
	}
	private int getCaretOffsetInternal() {
		synchronized (lock) {
			return caretOffset;
		}
	}
	public int getCaretOffset() {
		if (!isVisible())
			return 0;
		if (Thread.currentThread().equals(editorText.getDisplay().getThread())) {
			synchronized (this) {
				if (editorText.isDisposed()) {
					setCaretOffsetInternal(0);
				} else {
					setCaretOffsetInternal(editorText.getCaretOffset());
				}
			}
		} else {
			editorText.getDisplay().syncExec(runGetCaretOffset);
		}
		return getCaretOffsetInternal();
	}
	
	public synchronized boolean isModified() {
		return modified;
	}
	public synchronized void setModified(boolean modified) {
		ModificationEvent e
			= new ModificationEvent(this, this.getEditorTextWidget(), modified);
		fireEditorPreModified(e);
		
		this.modified = modified;
		
		fireEditorPostModified(e);
		
	}
	public void addModificationListener(IModificationListener listener) {
		this.modListeners.add(listener);
	}	
	public void removeModificationListener(IModificationListener listener) {
		this.modListeners.remove(listener);
	}
	protected void fireEditorPostModified(ModificationEvent e) {
		for(int i=0; i<this.modListeners.size(); i++) {
			((IModificationListener)modListeners.get(i)).objectModified(e);
		}
	}
	protected void fireEditorPreModified(ModificationEvent e) {
		for(int i=0; i<this.modListeners.size(); i++) {
			((IModificationListener)modListeners.get(i)).objectAboutToBeModified(e);
		}
	}
	
	public void setFont(Font font) {
		super.setFont(font);
		if (editorText != null) {
			editorText.setFont(font);
			lineNumbers.setFont(font);
		}
	}
	public void setSecondaryFont(Font font) {
		//default implementation specifies no action
		//this method exists for subclasses to be able to set
		//a secondary font of choice without having to cast the class instance
	}
	public Font getSecondaryFont() {
		//default implementation specifies no secondary font, so
		//this method returns null
		//subclasses should override this method to be able to
		//retrieve a secondary font that was set by setSecondaryFont
		return null;
	}
	
	public synchronized IHighlightingManager getHighlightingManager() {
		return highlightingManager;
	}
	public synchronized void setHighlightingManager(
				IHighlightingManager highlightingManager
			) {
		this.highlightingManager = highlightingManager;
	}
	
	private void setWindowVisible(boolean isVisible) {
		synchronized (lock) {
			this.isVisible = isVisible;
		}
	}
	private boolean getWindowVisible() {
		synchronized (lock) {
			return isVisible;
		}
	}
	public boolean isVisible() {
		if (editorText == null || editorText.isDisposed() || KenyaUI.isExiting())
			return false;
		
		if (Thread.currentThread().equals(editorText.getDisplay().getThread())) {
			synchronized (this) {
				setWindowVisible(this.getParent().isVisible());
			}
		} else {
			editorText.getDisplay().syncExec(runIsVisible);
		}
		return getWindowVisible();
	}
	
	public synchronized IButtonManager getButtonManager() {
		return buttonManager;
	}
	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent e) {
		// as part of a GUI it is anticipated that certain features may
		// need to react to ActionEvents.
		// the default implementation of this does sod-all so subclasses should
		// override this method to receive events
	}
	
}
