/*
 * Created on 28-Oct-2004
 */
package kenya.sole.ui.editor;

import kenya.interpreter.util.InterpreterInspectorContentProvider;
import kenya.interpreter.util.InterpreterInspectorListContentProvider;
import kenya.sole.ui.core.EditingWindow;
import kenya.sole.ui.editor.interpreter.CarrotPainter;
import kenya.sole.ui.editor.interpreter.InterpreterContinueButtonSelectionListener;
import kenya.sole.ui.editor.interpreter.InterpreterInspectorLabelProvider;
import kenya.sole.ui.editor.interpreter.InterpreterInspectorListLabelProvider;
import kenya.sole.ui.editor.interpreter.InterpreterInterpretSelectionListener;
import kenya.sole.ui.editor.interpreter.InterpreterMouseListener;
import kenya.sole.ui.editor.interpreter.InterpreterStepModeButtonSelectionListener;
import kenya.sole.ui.event.DefaultPaintListener;
import kenya.sourceCodeInformation.interfaces.IFunction;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import kenya.ui.IButtonManager;
import mediator.stackMachine.IStackMachineInformationProvider;

import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.scope.IMethodScope;

/**
 * result of abstracting the interpreter parts from EditingWindow
 * This EditorPart contains methods and state information to
 * analyse the execution of a Java program.
 * @author Thomas Timbul
 */
public class InterpreterPart extends EditorPart {
	
	private volatile Button interpreterInterpretButton;
	
	private volatile StyledText interpreterStdOutText;
	
	private volatile Text interpreterStdInText;
	
	private volatile Button interpreterTrackButton;
	
	private volatile Button interpreterEOFButton;
	
	private volatile Button interpreterTerminateButton;
	
	private volatile Button interpreterStepModeButton;
	
	private volatile boolean interpreterStepMode = false;
	
	private volatile boolean interpreterTrackMode = false;
	
	private volatile Button interpreterContinueButton;
	
	private volatile TreeViewer interpreterInspector;
	
	private volatile ListViewer interpreterList;
	
	private volatile Text interpreterArgsText;
	
	private volatile Label interpreterArgsLabel;
	
	private String[] interpreterArgsResult;
	
	private volatile CarrotPainter carrotPainter;
	
	private volatile IStackMachineInformationProvider baseStackMachine;
	
	private volatile StackMachine stackMachine;
	
	
	public InterpreterPart(
				Composite parent,
				int style,
				IStatusLineManager statusLine,
				IButtonManager buttonManager
			) {
		super(parent, style, statusLine, buttonManager);
	}
	
	/* (non-Javadoc)
	 * @see kenya.gui.editor.EditorPart#init()
	 */
	public void init() {
		super.init();
		
		carrotPainter = new CarrotPainter(this);
		
		editorText.addPaintListener(carrotPainter);
		InterpreterMouseListener iml = new InterpreterMouseListener(this);
		editorText.addMouseListener(iml);
		
		
		SashForm inspectorSash = new SashForm(this, SWT.HORIZONTAL | SWT.BORDER);

		interpreterList = new ListViewer(inspectorSash, SWT.SINGLE
				| SWT.V_SCROLL | SWT.H_SCROLL);
		interpreterList
				.setContentProvider(new InterpreterInspectorListContentProvider());
		interpreterList
				.setLabelProvider(new InterpreterInspectorListLabelProvider(
						this));
		interpreterList
				.addSelectionChangedListener(new ISelectionChangedListener() {

					private Object firstElement = null;

					public void selectionChanged(SelectionChangedEvent event) {
						IStructuredSelection selection = (IStructuredSelection) event
								.getSelection();
						if (selection.getFirstElement() != firstElement) {
							getInterpreterInspector().setInput(
									selection.getFirstElement());
							firstElement = selection.getFirstElement();
						}

						if (selection.getFirstElement() instanceof IMethodScope) {
							IFunction function = getBaseStackMachine()
									.lookupFunction(
											((IMethodScope) selection
													.getFirstElement())
													.getCurrentMethod()
													.getName());
							StyledText text = getEditorTextWidget();
							int offset = text.getOffsetAtLine(function
									.getPosition().getLineNumber() - 1);
							offset += function.getPosition().getColumnNumber() - 1;
							text.setCaretOffset(offset);
							text.setSelection(offset, offset
									+ function.getPosition().getTokenLength());
						}
					}
				});

		interpreterInspector = new TreeViewer(inspectorSash, SWT.SINGLE
				| SWT.V_SCROLL | SWT.H_SCROLL);
		interpreterInspector
				.setContentProvider(new InterpreterInspectorContentProvider());
		interpreterInspector
				.setLabelProvider(new InterpreterInspectorLabelProvider(this));
		interpreterInspector.setUseHashlookup(true);

		Composite ioComposite = new Composite(this, SWT.BORDER);
		FormLayout fl = new FormLayout();
		ioComposite.setLayout(fl);

		interpreterInterpretButton = new Button(ioComposite, SWT.PUSH);
		interpreterInterpretButton.setText("&Run");
		interpreterInterpretButton.setEnabled(false);

		interpreterStdOutText = new StyledText(ioComposite, SWT.MULTI
				| SWT.WRAP | SWT.V_SCROLL | SWT.BORDER);
		interpreterStdOutText.setText("");
		interpreterStdOutText.setEditable(false);
		interpreterStdOutText.setTabs(8);

		interpreterStdInText = new Text(ioComposite, SWT.SINGLE | SWT.BORDER);
		interpreterStdInText.setText("");
		interpreterStdInText.setEditable(true);
		interpreterStdInText.setEnabled(false);
		interpreterStdInText.setTabs(8);

		interpreterEOFButton = new Button(ioComposite, SWT.PUSH);
		interpreterEOFButton.setText("Input EOF");
		interpreterEOFButton.setEnabled(false);

		interpreterTerminateButton = new Button(ioComposite, SWT.PUSH);
		interpreterTerminateButton.setText("Terminate");
		interpreterTerminateButton.setEnabled(false);

		interpreterStepModeButton = new Button(ioComposite, SWT.TOGGLE);
		interpreterStepModeButton.setText("Step Mode");
		interpreterStepModeButton.setEnabled(false);
		interpreterStepModeButton.addSelectionListener(
					new InterpreterStepModeButtonSelectionListener(this)
				);
		setInterpreterStepMode(interpreterStepModeButton.getSelection());

		interpreterContinueButton = new Button(ioComposite, SWT.PUSH);
		interpreterContinueButton.setText("Continue");
		interpreterContinueButton.setEnabled(false);

		interpreterContinueButton.addSelectionListener(
					new InterpreterContinueButtonSelectionListener(this)
				);

		interpreterTrackButton = new Button(ioComposite, SWT.TOGGLE);
		interpreterTrackButton.setText("Track");
		interpreterTrackButton.setEnabled(true);
		interpreterTrackButton.addSelectionListener(new SelectionAdapter() {

			public void widgetSelected(SelectionEvent e) {
				int pri = Thread.currentThread().getPriority();
				Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
				setInterpreterTrackMode(interpreterTrackButton.getSelection());
				Thread.currentThread().setPriority(pri);
			}
		});
		interpreterTrackButton.setSelection(true);
		setInterpreterTrackMode(interpreterTrackButton.getSelection());

		interpreterArgsLabel = new Label(ioComposite, SWT.LEFT);
		interpreterArgsLabel.setText("Arguments");

		interpreterArgsText = new Text(ioComposite, SWT.SINGLE | SWT.BORDER);

		// top left
		FormData fd = new FormData();
		fd.top = new FormAttachment(0, 0);
		fd.left = new FormAttachment(0, 1);
		fd.right = new FormAttachment(interpreterTrackButton, -5);
		interpreterInterpretButton.setLayoutData(fd);

		// top middle left left
		fd = new FormData();
		fd.top = new FormAttachment(0, 0);
		fd.right = new FormAttachment(interpreterContinueButton, -5);
		fd.width = 100;
		fd.bottom = new FormAttachment(interpreterInterpretButton, 0,
				SWT.BOTTOM);
		interpreterTrackButton.setLayoutData(fd);

		// top middle left
		fd = new FormData();
		fd.top = new FormAttachment(0, 0);
		fd.right = new FormAttachment(interpreterStepModeButton, -5);
		fd.width = 100;
		fd.bottom = new FormAttachment(interpreterInterpretButton, 0,
				SWT.BOTTOM);
		interpreterContinueButton.setLayoutData(fd);

		// top middle right
		fd = new FormData();
		fd.top = new FormAttachment(0, 0);
		fd.right = new FormAttachment(interpreterTerminateButton, -5);
		fd.width = 100;
		fd.bottom = new FormAttachment(interpreterInterpretButton, 0,
				SWT.BOTTOM);
		interpreterStepModeButton.setLayoutData(fd);

		// top right
		fd = new FormData();
		fd.top = new FormAttachment(0, 0);
		fd.right = new FormAttachment(100, -1);
		fd.width = 100;
		fd.bottom = new FormAttachment(interpreterInterpretButton, 0,
				SWT.BOTTOM);
		interpreterTerminateButton.setLayoutData(fd);

		// middle
		fd = new FormData();
		fd.top = new FormAttachment(interpreterInterpretButton, 5);
		fd.bottom = new FormAttachment(interpreterStdInText, -5);
		fd.left = new FormAttachment(0, 1);
		fd.right = new FormAttachment(100, -1);
		interpreterStdOutText.setLayoutData(fd);

		// bottom left
		fd = new FormData();
		fd.left = new FormAttachment(0, 1); // 0%, 0 offset
		fd.bottom = new FormAttachment(interpreterArgsText, -5);
		fd.right = new FormAttachment(interpreterEOFButton, -5);
		interpreterStdInText.setLayoutData(fd);

		// bottom right
		fd = new FormData();
		fd.right = new FormAttachment(100, -1); // 100%, 0 offset
		fd.bottom = new FormAttachment(interpreterArgsText, -5);
		fd.width = 100;
		fd.top = new FormAttachment(interpreterStdInText, 0, SWT.TOP);
		interpreterEOFButton.setLayoutData(fd);

		fd = new FormData();
		fd.left = new FormAttachment(0, 1);
		fd.bottom = new FormAttachment(100, -5);
		fd.top = new FormAttachment(interpreterArgsText, 5, SWT.TOP);
		interpreterArgsLabel.setLayoutData(fd);

		fd = new FormData();
		fd.right = new FormAttachment(100, -1);
		fd.left = new FormAttachment(interpreterArgsLabel, 5);
		fd.bottom = new FormAttachment(100, -5);
		interpreterArgsText.setLayoutData(fd);

		SelectionListener interpretButtonListener
				= new InterpreterInterpretSelectionListener(this);
		interpreterInterpretButton
				.addSelectionListener(interpretButtonListener);
		
		this.setWeights(new int[] { 1, 1, 1 });
		
	}
	
	/* (non-Javadoc)
	 * @see kenya.gui.editor.EditorPart#setFont(org.eclipse.swt.graphics.Font)
	 */
	public void setFont(Font font) {
		super.setFont(font);
		if (interpreterInspector != null) {
			interpreterInspector.getControl().setFont(font);
			interpreterList.getControl().setFont(font);
		}
	}
	
	/* (non-Javadoc)
	 * @see kenya.gui.editor.EditorPart#setSecondaryFont(org.eclipse.swt.graphics.Font)
	 */
	public void setSecondaryFont(Font font) {
		if (getEditorTextWidget() != null) {
			interpreterStdOutText.setFont(font);
			interpreterStdInText.setFont(font);
		}
	}
	
	/*
	 *  (non-Javadoc)
	 * @see kenya.gui.editor.EditorPart#getSecondaryFont()
	 */
	public Font getSecondaryFont() {
		if(getEditorTextWidget()!=null) {
			return interpreterStdInText.getFont();
		}
		return null;
	}
	
	/* (non-Javadoc)
	 * @see kenya.gui.editor.EditorPart#buildEditorContent(org.eclipse.swt.custom.SashForm)
	 */
	protected void buildEditorContent(SashForm parent) {
		super.buildEditorContent(parent);
		
		editorText.setEditable(false);
		editorText.addPaintListener(new DefaultPaintListener(this));
	}
	
	public synchronized Button getInterpreterInterpretButton() {
		return interpreterInterpretButton;
	}
	
	public synchronized StyledText getInterpreterStdOutText() {
		return interpreterStdOutText;
	}
	
	public synchronized Text getInterpreterStdInText() {
		return interpreterStdInText;
	}
	
	public synchronized Button getInterpreterEOFButton() {
		return interpreterEOFButton;
	}
	
	public synchronized Button getInterpreterTerminateButton() {
		return interpreterTerminateButton;
	}
	
	public synchronized Button getInterpreterStepModeButton() {
		return interpreterStepModeButton;
	}

	public synchronized Button getInterpreterContinueButton() {
		return interpreterContinueButton;
	}

	public synchronized StackMachine getStackMachine() {
		return stackMachine;
	}

	public void setStackMachine(final StackMachine stackMachine) {
		if (Thread.currentThread()
				.equals(
						getInterpreterInspector().getControl().getDisplay()
								.getThread())) {
			synchronized (this) {
				this.stackMachine = stackMachine;
				interpreterInspector.setInput(stackMachine);
				interpreterList.setInput(stackMachine);
			}
		} else {
			getInterpreterInspector().getControl().getDisplay().syncExec(
					new Runnable() {

						public void run() {
							setStackMachine(stackMachine);
						}
					});
		}
	}
	
	public synchronized Button getInterpreterTrackButton() {
		return interpreterTrackButton;
	}
	
	public boolean getInterpreterTrackMode() {
		synchronized (lock) {
			return interpreterTrackMode;
		}
	}
	
	private void setInterpreterTrackMode(boolean interpreterTrackMode) {
		synchronized (lock) {
			this.interpreterTrackMode = interpreterTrackMode;
		}
	}
	
	public synchronized TreeViewer getInterpreterInspector() {
		return interpreterInspector;
	}
	
	public synchronized ListViewer getInterpreterList() {
		return interpreterList;
	}
	
	public synchronized Text getInterpreterArgsText() {
		return interpreterArgsText;
	}
	
	public String[] getInterpreterArgs() {
		if (Thread.currentThread().equals(
				getInterpreterArgsText().getDisplay().getThread())) {
			setInterpreterArgsResult(
					EditingWindow.parseIntoArgs(getInterpreterArgsText().getText())
				);
		} else {
			getInterpreterArgsText().getDisplay().syncExec(new Runnable() {

				public void run() {
					getInterpreterArgs();
				}
			});
		}
		return getInterpreterArgsResult();
	}
	
	private void setInterpreterArgsResult(String[] interpreterArgs) {
		synchronized (lock) {
			this.interpreterArgsResult = interpreterArgs;
		}
	}
	
	private String[] getInterpreterArgsResult() {
		synchronized (lock) {
			return interpreterArgsResult;
		}
	}
	
	public synchronized void setBaseStackMachine(
			IStackMachineInformationProvider smip) {
		baseStackMachine = smip;
	}
	
	public synchronized IStackMachineInformationProvider getBaseStackMachine() {
		return baseStackMachine;
	}
	
	public synchronized CarrotPainter getCarrotPainter() {
		return carrotPainter;
	}
	
	public void positionBlueCarrot(final ISourceCodeLocation loc) {
		if (Thread.currentThread().equals(
				getEditorTextWidget().getDisplay().getThread())) {
			
			synchronized (this) {
				int row = loc.getLineNumber() - 1;
				int col = loc.getColumnNumber() - 1;
				int offset = getEditorTextWidget().getOffsetAtLine(row)
						+ col;
				getCarrotPainter().setBlueCarrotOffset(offset);
				getInterpreterInspector().refresh();
			}
		} else {
			getEditorTextWidget().getDisplay().syncExec(new Runnable() {
				
				public void run() {
					positionBlueCarrot(loc);
				}
			});
		}
	}
	
	public void positionBlueCarrot(final int offset) {
		if (Thread.currentThread().equals(
				getEditorTextWidget().getDisplay().getThread())) {

			synchronized (this) {
				getCarrotPainter().setBlueCarrotOffset(offset);
				getInterpreterInspector().refresh();
			}
		} else {
			getEditorTextWidget().getDisplay().syncExec(new Runnable() {

				public void run() {
					positionBlueCarrot(offset);
				}
			});
		}
	}
	
	public void setInterpreterStepMode(boolean result) {
		synchronized (lock) {
			interpreterStepMode = result;
		}
	}
	
	public boolean getInterpreterStepMode() {
		synchronized (lock) {
			return interpreterStepMode;
		}
	}
	
}
