/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/
/*
 * Created on Jul 25, 2004
 *
 */
package kenya.sole.ui.editor.util;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import kenya.sole.ui.editor.EditorPart;
import kenya.sole.ui.editor.KenyaPart;
import kenya.sole.ui.util.AbstractHighlighter;
import kenya.sole.ui.util.IHighlightingManager;
import kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import kenya.sourceCodeInformation.interfaces.ISourceCodeInformation;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;

import org.eclipse.swt.custom.LineBackgroundEvent;
import org.eclipse.swt.custom.LineBackgroundListener;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Display;

/**
 * @author Matthew Sackman
 * @version 1
 */
public class DefaultHighlightingManager implements IHighlightingManager {

    public static final Color WHITE = new Color(Display.getDefault(), 255, 244,
            212);
    
    private final KenyaPart editor;
    
    private final ErrorHighlighter errorH;
    
    private final WarningHighlighter warningH;
    
    private final LinkedLocationHighlighter linkedLocationH;
    
    private final Map pointsToErrors = new HashMap();
    
    private final Map pointsToWarnings = new HashMap();
    
    public DefaultHighlightingManager(KenyaPart editor,
            ErrorHighlighter error, WarningHighlighter warning,
            LinkedLocationHighlighter linkedLocation) {
        this.editor = editor;
        this.errorH = error;
        this.warningH = warning;
        this.linkedLocationH = linkedLocation;
    }

    public ErrorHighlighter getErrorHighlighter() {
        return errorH;
    }

    public WarningHighlighter getWarningHighlighter() {
        return warningH;
    }

    public LinkedLocationHighlighter getLinkedLocationHighlighter() {
        return linkedLocationH;
    }

    public void paint(PaintEvent e) {
        errorH.paintControl(e);
        warningH.paintControl(e);
        linkedLocationH.paintControl(e);
    }

    public void clear() {
        synchronized (this) {
            errorH.clearHighlights();
            warningH.clearHighlights();
            linkedLocationH.clearHighlights();
            pointsToErrors.clear();
            pointsToWarnings.clear();
        }
        editor.selectInKenyaErrorTable(null);
    }

    public void addSourceCodeError(final ISourceCodeError error) {
        if (!editor.isVisible()) return;
        final StyledText text = editor.getEditorTextWidget();
        
        if (Thread.currentThread().equals(text.getDisplay().getThread())) {
            synchronized (this) {
                final ISourceCodeLocation location = error.getLocation();
                final int row = location.getLineNumber() - 1;
                final int column = location.getColumnNumber() - 1;
                final int length = location.getTokenLength();
                if (!editor.isVisible() || text.isDisposed()) return;
                int myRow = row;
                if (myRow < text.getContent().getLineCount() && myRow >= 0) {
                    int offset = text.getOffsetAtLine(myRow) + column;
                    errorH.addHighlight(offset, length, row);
                    Point point = new Point(offset, offset + length);
                    registerError(point, error);
                }
            }
        } else {
            text.getDisplay().syncExec(new Runnable() {

                public void run() {
                    addSourceCodeError(error);
                }
            });
        }
    }

    public void addSourceCodeWarning(final ISourceCodeInformation warning) {
        if (!editor.isVisible()) return;
        final StyledText text = editor.getEditorTextWidget();
        if (Thread.currentThread().equals(text.getDisplay().getThread())) {
            synchronized (this) {
                final ISourceCodeLocation location = warning.getLocation();
                final int row = location.getLineNumber() - 1;
                final int column = location.getColumnNumber() - 1;
                final int length = location.getTokenLength();

                if (!editor.isVisible() || text.isDisposed()) return;
                int myRow = row;
                if (myRow >= text.getLineCount())
                        myRow = text.getLineCount() - 1;
                int offset = text.getOffsetAtLine(myRow) + column;
                warningH.addHighlight(offset, length, row);
                Point point = new Point(offset, offset + length);
                registerWarning(point, warning);
            }
        } else {
            text.getDisplay().syncExec(new Runnable() {

                public void run() {
                    addSourceCodeWarning(warning);
                }
            });
        }
    }

    private void addHighlight(final ISourceCodeLocation location,
            final AbstractHighlighter highlighter) {
        if (!editor.isVisible()) return;

        final StyledText text = editor.getEditorTextWidget();

        if (Thread.currentThread().equals(text.getDisplay().getThread())) {
            synchronized (this) {
                final int row = location.getLineNumber() - 1;
                final int column = location.getColumnNumber() - 1;
                final int length = location.getTokenLength();
                if (!editor.isVisible() || text.isDisposed()) return;
                int myRow = row;
                if (myRow >= text.getLineCount())
                        myRow = text.getLineCount() - 1;
                int offset = text.getOffsetAtLine(myRow) + column;
                highlighter.addHighlight(offset, length, myRow);
            }
        } else {
            text.getDisplay().syncExec(new Runnable() {

                public void run() {
                    addHighlight(location, highlighter);
                }
            });
        }
    }

    public MouseListener createKenyaMouseListener(final EditorPart editor) {
        final MouseListener mouseListener = new MouseListener() {

            public void mouseDown(MouseEvent e) {
            }

            public void mouseDoubleClick(MouseEvent e) {
            }

            public void mouseUp(MouseEvent e) {
                if (e.button == 1) {
                    int offset = editor.getEditorTextWidget().getCaretOffset();
                    showLinkedHighlights(offset);
                }
            }
        };

        return mouseListener;
    }

    public void showLinkedHighlights(int offset) {
        ISourceCodeInformation error = searchForErrorOrWarning(offset);
        if (error != null) {
            linkedLocationH.clearHighlights();
            ISourceCodeLocation[] locations = error.getLinkedInformation();
            if (locations != null) {
                for (int idx = 0; idx < locations.length; idx++) {
                    addHighlight(locations[idx], linkedLocationH);
                }
            }
            editor.selectInKenyaErrorTable(error);
            editor.safeRedraw();
        }
    }

    private synchronized void registerError(Point location,
            ISourceCodeError error) {
        pointsToErrors.put(location, error);
    }

    private synchronized void registerWarning(Point location,
            ISourceCodeInformation warning) {
        pointsToWarnings.put(location, warning);
    }

    private synchronized ISourceCodeInformation searchForErrorOrWarning(
            int offset) {
        Iterator it = pointsToErrors.keySet().iterator();
        while (it.hasNext()) {
            Point point = (Point) it.next();
            if (point.x < offset && point.y >= offset) { return (ISourceCodeInformation) pointsToErrors
                    .get(point); }
        }

        it = pointsToWarnings.keySet().iterator();
        while (it.hasNext()) {
            Point point = (Point) it.next();
            if (point.x < offset && point.y >= offset) { return (ISourceCodeInformation) pointsToWarnings
                    .get(point); }
        }
        return null;
    }

    public LineBackgroundListener createKenyaLineBackgroundListener(
            final EditorPart editor) {
    	
			return new LineBackgroundListener() {
			
			    public void lineGetBackground(LineBackgroundEvent event) {
			        int lineNumber = editor.getLineNumberWidget()
			                .getLineAtOffset(event.lineOffset);
			        if (errorH.isHighlightOnLine(lineNumber))
			            event.lineBackground = errorH.getBackgroundColor();
			        else if (warningH.isHighlightOnLine(lineNumber))
			            event.lineBackground = warningH.getBackgroundColor();
			        else if (linkedLocationH.isHighlightOnLine(lineNumber))
			            event.lineBackground = linkedLocationH.getBackgroundColor();
			        else
			            event.lineBackground = WHITE;
			    }
			};
    }
}