#!/usr/bin/php
<?php function main() { ?>

<div class = "normal_text">
In this second part of the tutorial, we will be modelling a Map that associates unique keys with values (such as eg. <a href="http://java.sun.com/javase/6/docs/api/java/util/Map.html">Map</a> in Java) and show how Alloy can be used to explore dynamic behaviour.
<br/>
<br/>
The static aspects are easy to specify:
<br/>
<br/>
<span class = "code">
	<span class = "keyword">module</span> examples/tutorial/Map<br/><br/>
	<span class = "keyword">abstract</span> <span class = "keyword">sig</span> Object {}<br/>
	<span class = "keyword">sig</span> Key, Value <span class = "keyword">extends</span> Object {}<br/>
	<span class = "keyword">sig</span> Map { values: Key -> Value }</span>
<br/><br/>
As in Java, <span class = "keyword">abstract</span> ensures that there cannot be any direct 'instances' of <span class = "code">Object</span>. Similarly, <span class = "keyword">extends</span> means that <span class = "code">Objects</span> can either be <span class = "code">Keys</span> or <span class = "code">Values</span>, but not both.
<br/><br/>
<span class = "code">values</span> above is a <span class = "emphasise_text">relation</span> that represents the mapping of <span class = "code">Keys</span> to <span class = "code">Values</span>. This would be written in a schema in Object-Z as <span class = "code">values &sube; K &times; V</span> where K is the set of all Keys and V is the set of all Values.
<br/>
<br/>
We can visualize this model as before:
<br/>
<br/>
<span class = "code">
	<span class = "keyword">pred</span> show() {}
	<span class = "keyword">run</span> show <span class = "keyword">for</span> 2
</span>
<br/>
The first instance produced by Alloy only consists of a <span class = "code">Key</span> and a <span class = "code">Value</span> but no <span class = "code">Map</span>. We could browse through the various solutions until we find one that actually includes a <span class = "code">Map</span>, however there is a better way.
<br/>
As mentioned before, Alloy produces instances that satisfy the predicate passed to <span class = "keyword">run</span>. We can add a constraint that specifies that there is at least one <span class = "code">Map</span>:
<br/>
<br/>
<span class = "code">
	<span class = "keyword">pred</span> show() { #Map > 0 }<br/>
	<span class = "keyword">run</span> show <span class = "keyword">for</span> 2
</span>
<br/>
We now get 2 <span class = "code">Maps</span> and one Value. To get exactly one <span class = "code">Map</span>, we can either change the constraint to
<br/>
<br/>
<span class = "code">
	#Map = 1
</span>
<br/>
or we can modify the <span class = "keyword">run</span> command as follows:
<br/>
<br/>
<span class = "code">
	<span class = "keyword">run</span> show <span class = "keyword">for</span> 2 but <span class = "keyword">exactly</span> 1 Map
</span>
<br/>
Next, let us check that the mapping of keys to values is unique. This can be done in Alloy via <span class = "emphasise_text">assertions</span>:
<br/>
<br/>
<span class = "code">
	<span class = "keyword">assert</span> mappingIsUnique {<br/>
	&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class = "keyword">all</span> m:Map, k:Key, v, v':Value |<br/>
	&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;k -> v <span class = "keyword">in</span> m.values <span class = "keyword">and</span> k -> v' <span class = "keyword">in</span> m.values <span class = "keyword">implies</span> v = v'<br/>
	}
</span>
<br/>
This says that if a <span class = "code">Map</span> <span class = "code">m</span> maps a key <span class = "code">k</span> to two values <span class = "code">v</span> and <span class = "code">v'</span> then they must be the same. Note how the relational product operator ("<span class = "code">-></span>") in <span class = "code">k -> v</span> is used to represent the tuple (k, v) and how <span class = "code">m.values</span> is treated as a set of tuples <span class = "code">{(Key, Value)}</span>. To check an assertion, we can use the command <span class = "keyword">check</span>. As for <span class = "keyword">run</span>, we have to specify the scope:
<br/>
<br/>
<span class = "code">
	<span class = "keyword">check</span> mappingIsUnique <span class = "keyword">for</span> 2
</span>
<br/>
If we execute this, Alloy searches all possible combinations of at most two objects of each signature for a counterexample that violates the assertion. Since it cannot find one, it tells us that the assertion may be valid. However, if we increase the scope to three, it produces the following:
<br/>
<br/>
	<center>
		<img src="Images/Example_2_01.jpg" align="center"/>
	</center>
<br/>
To make this even more obvious, we can ask Alloy to project over <span class = "code">Map</span> using the Projection button:
<br/>
<br/>
	<center>
		<img src="Images/Example_2_02.jpg" align="center"/>
	</center>
<br/>
<br/>
We see that a key refers to two values. To fix this, we use the <span class = "keyword">lone</span> keyword that we have already seen:
<br/>
<br/>
<span class = "code">
	<span class = "keyword">sig</span> Map { values: Key -> <span class = "keyword">lone</span> Value }<br/>

</span>
<br/>
Now, the assertion holds even if we increase the scope to 15 (say). This produces a much larger coverage than any testcase ever could and gives us a high level of confidence that the assertion may be valid.
<br/>
<br/>
Finally, to make the instances of later steps less verbose, we include the constraint that all keys (and values) belong to some <span class = "code">Map</span>:
<br/>
<br/>
<span class = "code">
	<span class = "keyword">fact</span> {<br/>
	&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class = "keyword">all</span> k:Key | <span class = "keyword">some</span> v:Value, m:Map | k -> v in m.values<br/>
	&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class = "keyword">all</span> v:Value | <span class = "keyword">some</span> k:Key, m:Map | k -> v in m.values<br/>
	}
</span>
<br/>
Note that, unlike in the Queue example, it makes sense for a key/value to belong to more than one <span class = "code">Map</span>. Also observe that the fact above is anonymous; Alloy does not require you to provide a name.

<br/><br/>

We have now developed the static aspects of the <span class = "code">Map</span>. In the next section we will continue to develop the dynamics of our example.

</div>

<?php } include 'template.php'; ?>
