/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/



/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/


#ifndef SOFTSHELL_AGGREGATION_LOCALWORKITEM_H_INCLUDED
#define SOFTSHELL_AGGREGATION_LOCALWORKITEM_H_INCLUDED

#include "tools/common.cuh"

namespace Softshell
{
  class CombWorkpackageBase;
  class LocalAggregation;

  extern __constant__ uint LocalCombWpDataSize;
  class LocalCombWorkpackage
  {

    friend class CombWorkpackageBase;
    friend class LocalAggregation;
    __device__  volatile CombWorkpackageBase* getWpForThread(uint& i) volatile
    {
      if(i < selElements())
          return reinterpret_cast<volatile CombWorkpackageBase*>(this);
      else
        return 0;
    }
    __device__  volatile const CombWorkpackageBase* getWpForThread(uint& i) const volatile
    {
      if(i < selElements())
          return reinterpret_cast<volatile const CombWorkpackageBase*>(this);
      else
        return 0;
    }

    __device__ bool combineWith(CombWorkpackageBase* other)
    {
      return false;
    }
    __device__ CombWorkpackageBase* push_back(CombWorkpackageBase* other)
    {
      //dont do that!
      return 0;
    }
    __device__ uint structSize() const
    {
      return 0;
    }
    __device__ void paste(unsigned char* data) const
    {

    }
    __device__ uint spots()
    {
      return LocalCombWpDataSize/max(1,locked_execthreads&0xFFFF);
    }

     template<class TWorkItem>
    __device__ volatile TWorkItem* getWorkItem(uint i) volatile
    {
      if(i < selElements())
        return ((volatile TWorkItem*)data) + i;
      else
        return 0;
    }
    template<class TWorkItem>
    __device__ volatile TWorkItem* getWorkItem(uint i) const volatile
    {
      if(i < selElements())
        return ((volatile const TWorkItem*)data) + i;
      else
        return 0;
    }
    template<class TParams>
    __device__ TParams* getParams(uint i) volatile
    {
      if(i < selElements())
        return  (TParams*)(((uint*)this)-offset);
      else
        return 0;
    }
    template<class TParams>
    __device__ const TParams* getParams(uint i) const volatile
    {
      if(i < selElements())
        return (const TParams*)(((uint*)this)-offset);
      else
        return 0;
    }
    template<class TWorkItem>
    __device__ TWorkItem* addWorkItem(const TWorkItem& item)
    {
      uint old = atomicAdd(&entryPoint_elements,1)&0xFFFFF;
      if(old >= spots())
      {
        atomicSub(&entryPoint_elements,1);
        return 0;
      }
      TWorkItem* p = ((TWorkItem*)data) + old;
      *p = item;
      return p;
    }
    template<class TWorkItem>
    __device__ TWorkItem* addWorkItemUnsafe(const TWorkItem& item)
    {
      if(selElements() >= spots())
        return 0;

      TWorkItem* p = ((TWorkItem*)data) + selElements();
      *p = item;
      ++entryPoint_elements;
      return p;
    }
    __device__ void setElements(uint num)
    {
      entryPoint_elements = (entryPoint_elements & 0xFFFF0000) | num;
    }
    template<class TWorkItem>
    __device__ TWorkItem* getWorkItemsPointer()
    {
      return (TWorkItem*)data;
    }
    template<class TWorkItem>
    __device__ uint addElements(uint num, const TWorkItem* items)
    {
      int canAdd = spots() - selElements();
      canAdd = min(num,max(0, canAdd));
      TWorkItem* p = getWorkItemsPointer<TWorkItem>();
      for(int i = 0; i < canAdd; ++i)
        p[selElements()+i] = items[i];
      entryPoint_elements += canAdd;
      return canAdd;
    }
  public:

    //eventId and eventLaunchId must be first!
    uint eventId;
    uint eventLaunchId;
    uint locked_execthreads;
    uint paramSize;
    uint entryPoint_elements;
    uint offset;

    PointerEquivalent func;
    void* data;
    clock64_t timeout;

    __device__ uint selElements() const volatile
    {
      return entryPoint_elements & 0x0000FFFF;
    }

    __device__ uint numWorkItems() volatile
    {
      return entryPoint_elements & 0x0000FFFF;
    }
    __device__ uint numWorkPackages() volatile
    {
      return 1;
    }
    __device__ void destroy(uint CombWorkpackageBaseSize) volatile
    {
      //dont destroy, but rather allow for reuse
      entryPoint_elements &= 0xFFFF0000;
      __threadfence_block();
      locked_execthreads = 0x80000000 | (locked_execthreads & 0x4000FFFF);

      ////destroy if data points to global!
      //if(!isshared(data))
      //{
      //  free(((unsigned char*)data) - CombWorkpackageBaseSize - (paramSize-1)*sizeof(uint));
      //}
    }

    __device__ uint reserveElements(uint num, uint& offset)
    {
      return 0;
    }
    template<class Param>
    __device__ bool compareParam(const Param* param) const
    {
      if(paramSize != sizeof(Param)+1) return false;
      const uint* nP = (const uint*)(param);
      const uint* mP = (const uint*)(data) - (paramSize-1);
      for(uint i = 0; i < sizeof(Param)/sizeof(uint); ++i)
        if(*nP++ != *mP++) return false;
      return true;
    }
  };

}
#endif //SOFTSHELL_AGGREGATION_LOCALWORKITEM_INCLUDED
