/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/


//#ifndef SOFTSHELL_DISTIBUTOR_EVENTMANAGER_INCLUDED
#ifdef __CUDACC__
#include "distributor/eventmanager.cuh"
#endif
//#endif

#ifndef SOFTSHELL_API_DEVENT_H_INCLUDED
#define SOFTSHELL_API_DEVENT_H_INCLUDED

#include "tools/types.h"
#include "api/priority.h"

#ifdef __CUDACC__
#include "distributor/eventmanager.cuh"
#include "timing/timesync.cuh"
#endif

#include "cuda.h"

namespace Softshell
{
  class DScheduler;
  class Workpackage;
  class EventManager;
  class DEvent
  {
    friend class DScheduler;
    friend class Workpackage;
    friend class EventManager;
    friend class Aggregation;
    friend class LocalAggregation;
    uint eventId;
    uint eventLaunchId;

  public:
    __device__ DEvent(uint _eventId, uint _eventLaunchId) : eventId(_eventId), eventLaunchId(_eventLaunchId) { }
 #ifdef __CUDACC__
    __device__ DEvent(const DEvent &other) : eventId(other.eventId), eventLaunchId(other.eventLaunchId) { }
    __device__ DEvent(volatile const DEvent &other) : eventId(other.eventId), eventLaunchId(other.eventLaunchId) { }

    __device__ uint getEventId() const
    {
      return eventId;
    }
    __device__ uint getEventLaunchId() const
    {
      return eventLaunchId;
    }

    __device__ Priority getPriority() const
    {
      return d_eventManager.getPriority(eventId);
    }

    __device__ float getDeadline() const
    {
      return d_eventManager.getDeadline(eventId);
    }

    __device__ float getPeriodicity() const
    {
      return d_eventManager.getPeriodicity(eventId);
    }

    __device__ void setPriority(Priority priority)
    {
      return d_eventManager.setPriority(eventId, priority);
    }

    __device__ void setDeadline(float deadline)
    {
      d_eventManager.setDeadline(eventId, deadline);
    }

    __device__ void setPeriodicity(float periodicity)
    {
      return d_eventManager.setPeriodicity(eventId, periodicity);
    }



    __device__ double launchTime() const
    {
      return d_eventManager.getLaunchTime(eventLaunchId);
    }

    __device__ double remainingTime() const
    {
      float deadline = d_eventManager.getDeadline(eventId);
      if(deadline == 0) return INF_TIME;
      double thisdeadline = d_eventManager.getLaunchTime(eventLaunchId) + deadline;
      return thisdeadline - getTime();
    }

    __device__ uint activeWorkpackages() const
    {
      return d_eventManager.getActiveWorkpackages(eventLaunchId);
    }

    __device__ uint processedWorkpackages() const
    {
      return d_eventManager.getProcessedWorkpackages(eventLaunchId);
    }


    __device__ bool equalLaunch(DEvent const & other) const
    {
      return eventId == other.eventId &&
             eventLaunchId == other.eventLaunchId;
    }
    __device__ bool operator != (DEvent const & other) const
    {
      return eventId != other.eventId;
    }
    __device__ bool operator == (DEvent const & other) const
    {
      return eventId == other.eventId;
    }
#endif
  };
};


#endif //SOFTSHELL_API_DEVENT_INCLUDED
