/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*   Rostislav Khlebnikov /  khlebnikov ( at ) icg.tugraz.at
*/


#ifndef SOFTSHELL_API_EVENT_H_INCLUDED
#define SOFTSHELL_API_EVENT_H_INCLUDED

#include "tools/types.h"
#include "api/priority.h"
#include "distributor/device.h"
#include "distributor/deviceentries.h"
#include "distributor/deviceentries.cuh"
#include "distributor/eventworkpackagemanagement.h"
#include "distributor/hscheduler.h"

#include <vector>
#include <memory>

// define override keyword for old g++ versions (no functionality of course!)
#ifdef __GNUC__
#if __GNUC_PREREQ(4,7)
#else
#define override
#endif
#endif


namespace Softshell
{
    struct EventStatistics
    {
        uint id;
        uint processedWorkpackages;
        uint activeWorkpackages;
        double runningTime;
        double expectedCompletion;
    };

    class EventBase
    {
    public:
        virtual void done(uint launchId, Device* device, float time) = 0;
        virtual uint getId() const = 0;

        virtual ~EventBase() {}
    };

    class EventNotifier
    {
    public:
        virtual void completed(EventBase* event, uint launchId, float timetaken, void* data) = 0;
        virtual  ~EventNotifier() { }
    };

    class DevicePointerList;
    template<class TEventCreatorDestroyer>
    class EventImpl;

    template<class TEventCreatorDestroyer>
    class Event : public EventBase
    {
    public:
        typedef typename TEventCreatorDestroyer::CreatorParamsType CreatorParamsType;
        friend class Scheduler;

    private:
        EventImpl<TEventCreatorDestroyer>* pImpl;

        DevicePointerList* devicepointerlist;

        Event(HScheduler* scheduler);
    public:
        void join();
        void join(uint id);
        void cancel();
        void cancel(uint id);

        uint getNumActiveLaunches() const;
        std::vector<EventStatistics> getEventStatus() const;

        void setPeriodicity(float periodicity);
        void setDeadline(float deadline);
        void setPriority(Priority priority);

        void registerNotifier(EventNotifier* notifier);
        void unregisterNotifier(EventNotifier* notifier);

        double getExecTime(uint id);

        uint fire(const CreatorParamsType& creatorParams, Device* device = nullptr);

        virtual void done(uint launchId, Device* device, float time) override;
        virtual uint getId() const override;
        virtual ~Event();

    };

    class PEvent_Interface
    {
      friend class PEvent;
    protected:
      static uint& getLastTypeCounter();
      template<typename CreatorParamsType>
      static uint getTypeId();
    public:
      virtual void join() = 0;
      virtual void join(uint id) = 0;
      virtual void cancel() = 0;
      virtual void cancel(uint id) = 0;

      virtual uint getNumActiveLaunches() const = 0;
      virtual std::vector<EventStatistics> getEventStatus() const = 0;

      virtual void setPeriodicity(float periodicity) = 0;
      virtual void setDeadline(float deadline) = 0;
      virtual void setPriority(Priority priority) = 0;

      virtual void registerNotifier(EventNotifier* notifier) = 0;
      virtual void unregisterNotifier(EventNotifier* notifier) = 0;

      virtual double getExecTime(uint id) = 0;

      virtual uint fire(const void* creatorParams, uint creatorParamsTypeId, Device* device = nullptr) = 0;

      virtual PEvent_Interface* clone() const = 0;
      virtual ~PEvent_Interface() { }
    };

    template<class TEventCreatorDestroyer>
    class PEvent_Implementation : public PEvent_Interface
    {
      uint myid;
#ifndef WIN32
      Event<TEventCreatorDestroyer> mutable * event;
    public:
      PEvent_Implementation(Event<TEventCreatorDestroyer>* event);
#else
      std::shared_ptr<Event<TEventCreatorDestroyer> > event;
    public:
      PEvent_Implementation(std::shared_ptr<Event<TEventCreatorDestroyer> > event);
#endif

      void join() { event->join(); }
      void join(uint id) { event->join(id); }
      void cancel() { event->cancel(); }
      void cancel(uint id) { event->cancel(id); }

      uint getNumActiveLaunches() const { return event->getNumActiveLaunches(); }
      std::vector<EventStatistics> getEventStatus() const { return event->getEventStatus(); }

      void setPeriodicity(float periodicity) { event->setPeriodicity(periodicity); }
      void setDeadline(float deadline) { event->setDeadline(deadline); }
      void setPriority(Priority priority) { event->setPriority(priority); }

      void registerNotifier(EventNotifier* notifier)  { event->registerNotifier(notifier); }
      void unregisterNotifier(EventNotifier* notifier) { event->unregisterNotifier(notifier); }

      double getExecTime(uint id) { return event->getExecTime(id); }

      uint fire(const void* creatorParams, uint creatorParamsTypeId, Device* device = nullptr);

      PEvent_Interface* clone() const;
      ~PEvent_Implementation();
    };

   class PEvent
    {
      PEvent_Interface* thePEvent;
      const PEvent& operator = (const PEvent& );
    public:
      PEvent() : thePEvent(nullptr) { }
      PEvent(const PEvent_Interface& pevent) : thePEvent(pevent.clone()) { }
      PEvent(const PEvent& pevent) : thePEvent(pevent.thePEvent->clone()) { }
      const PEvent& operator = (const PEvent_Interface& pevent)
      {
        delete thePEvent;
        thePEvent = pevent.clone();
        return *this;
      }
#ifndef WIN32
      template<class TEventCreatorDestroyer>
      PEvent(Event<TEventCreatorDestroyer>* event) : thePEvent(new PEvent_Implementation<TEventCreatorDestroyer>(event)) { }
      template<class TEventCreatorDestroyer>
      const PEvent& operator = (Event<TEventCreatorDestroyer>* event)
      {
        delete thePEvent;
        thePEvent = new PEvent_Implementation<TEventCreatorDestroyer>(event);
        return *this;
      }
#else
      template<class TEventCreatorDestroyer>
      PEvent(std::shared_ptr<Event<TEventCreatorDestroyer> > event) : thePEvent(new PEvent_Implementation<TEventCreatorDestroyer>(event)) { }

      template<class TEventCreatorDestroyer>
      const PEvent& operator = (std::shared_ptr<Event<TEventCreatorDestroyer> > event)
      {
        delete thePEvent;
        thePEvent = new PEvent_Implementation<TEventCreatorDestroyer>(event);
        return *this;
      }
#endif
      void join() { thePEvent->join(); }
      void join(uint id) { thePEvent->join(id); }
      void cancel() { thePEvent->cancel(); }
      void cancel(uint id) { thePEvent->cancel(id); }

      uint getNumActiveLaunches() const { return thePEvent->getNumActiveLaunches(); }
      std::vector<EventStatistics> getEventStatus() const { return thePEvent->getEventStatus(); }

      void setPeriodicity(float periodicity) {  thePEvent->setPeriodicity(periodicity); }
      void setDeadline(float deadline) {  thePEvent->setDeadline(deadline); }
      void setPriority(Priority priority) {  thePEvent->setPriority(priority); }

      void registerNotifier(EventNotifier* notifier) {  thePEvent->registerNotifier(notifier); }
      void unregisterNotifier(EventNotifier* notifier) {  thePEvent->unregisterNotifier(notifier); }

      double getExecTime(uint id) { return thePEvent->getExecTime(id); }

      template<class CreatorParamsType>
      uint fire(const CreatorParamsType& creatorParams, Device* device = nullptr) { return thePEvent->fire(&creatorParams, PEvent_Interface::getTypeId<CreatorParamsType>(), device); }

      //this takes care of the compatiblity between Windows and Linux ... and is really ugly
      PEvent* operator -> () { return this; }
      const PEvent* operator -> () const { return this; }
      PEvent& operator * ()  {return *this; }
      const PEvent& operator * () const {return *this; }

      ~PEvent() { if(thePEvent) delete thePEvent; }
    };
};

#include "distributor/event.ipp"

#endif //SOFTSHELL_API_EVENT_INCLUDED
