/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*   Rostislav Khlebnikov /  khlebnikov ( at ) icg.tugraz.at
*/


#ifndef SOFTSHELL_API_SCHEDULER_H_INCLUDED
#define SOFTSHELL_API_SCHEDULER_H_INCLUDED

#include <vector>
#include <memory>
#include "flags.h"
#include "tools/types.h"
#include "event.h"
#include "tools/configcollection.h"


namespace Softshell
{
  class Device;
  class HScheduler;
  class ControllerInfo;
  class Scheduler
  {
    HScheduler* hscheduler;
    DevicePointerList* devicepointerlist;
    typedef std::vector<uint> DeviceIDListType;
    DeviceIDListType deviceid_list;

    ConfigCollection config;

    template<class EventCreatorDestroyer>
    void registerEventCreatorDestroyer();
  public:
    Scheduler(int device_id = ANY_DEVICE, const Flags& flags = Flags() );
    Scheduler(std::vector<uint> const & device_ids, const Flags& flags = Flags());
    Scheduler(const ControllerInfo& info, int device_id = ANY_DEVICE, const Flags& flags = Flags() );
    Scheduler(const ControllerInfo& info, std::vector<uint> const & device_ids, const Flags& flags = Flags() );

    bool init(int device_id = ANY_DEVICE, const Flags& flags = Flags());
    bool init(std::vector<uint> const & device_ids, const Flags& flags = Flags());
    bool init(const ControllerInfo& info, int device_id = ANY_DEVICE, const Flags& flags = Flags());
    bool init(const ControllerInfo& info, std::vector<uint> const & device_ids, const Flags& flags = Flags());

    bool is_init();

#ifdef __CUDACC__
    template <class TWorkpackage, class TProcedure>
#ifndef WIN32
    Event<DefaultEventCreator<TWorkpackage, TProcedure, typename TWorkpackage::ParamsType> > * createEvent();
#else
    std::shared_ptr<Event<DefaultEventCreator<TWorkpackage, TProcedure, typename TWorkpackage::ParamsType> > > createEvent();
#endif
#endif

    template <class TEventCreatorDestroyer>
#ifndef WIN32
    Event<TEventCreatorDestroyer> * createEvent();
#else
    std::shared_ptr<Event<TEventCreatorDestroyer> > createEvent();
#endif

    Device* getDevice(int device_id);
    std::vector<Device*> getDevices();

    void pause();
    void start();
    void shutdown();

    template<class T>
    void setConfigParam(const std::string& name, T val);
    template<class T>
    T getConfigParam(const std::string& name);

    void setController(const ControllerInfo& controllerInfoy);
  };

};

namespace Softshell
{
#ifdef __CUDACC__
  template<class EventCreatorDestroyer>
  void Scheduler::registerEventCreatorDestroyer()
  {
    for( Scheduler::DeviceIDListType::const_iterator it = deviceid_list.begin(); it != deviceid_list.end(); ++it)
      devicepointerlist->getCreatorFunc<EventCreatorDestroyer>(*it),
      devicepointerlist->getDestroyerFunc<EventCreatorDestroyer>(*it);
  }
#endif

#ifdef __CUDACC__
  template <class TWorkpackage, class TProcedure>
#ifndef WIN32
  Event<DefaultEventCreator<TWorkpackage, TProcedure, typename TWorkpackage::ParamsType> > * Scheduler::createEvent()
#else
  std::shared_ptr<Event<DefaultEventCreator<TWorkpackage, TProcedure, typename TWorkpackage::ParamsType> > > Scheduler::createEvent()
#endif
  {
    return createEvent<DefaultEventCreator<TWorkpackage, TProcedure, typename TWorkpackage::ParamsType> >();
  }
#endif
  template <class TEventCreatorDestroyer>
#ifndef WIN32
  Event<TEventCreatorDestroyer> * Scheduler::createEvent()
#else
  std::shared_ptr<Event<TEventCreatorDestroyer> > Scheduler::createEvent()
#endif
  {
    registerEventCreatorDestroyer<TEventCreatorDestroyer>();

    Event<TEventCreatorDestroyer> *newevent = new Event<TEventCreatorDestroyer>(hscheduler);
#ifndef WIN32
    return newevent;
#else
    return std::shared_ptr<Event<TEventCreatorDestroyer> >(newevent);
#endif
  }

  template<class T>
  void Scheduler::setConfigParam(const std::string& name, T val)
  {
    config.set(name, val);
  }
  template<class T>
  T Scheduler::getConfigParam(const std::string& name)
  {
    return config.get<T>(name);
  }
}

#endif //SOFTSHELL_API_SCHEDULER_INCLUDED
