/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*   Rostislav Khlebnikov /  khlebnikov ( at ) icg.tugraz.at
*/

#ifndef SOFTSHELL_API_WORKPACKAGE_H_INCLUDED
#define SOFTSHELL_API_WORKPACKAGE_H_INCLUDED

#include "tools/types.h"
#include "tools/common.cuh"
#include "api/devent.h"

#ifdef __CUDACC__
#include "aggregation/localworkitem.cuh"
#include "memory/roundrobinalloc.cuh"
#endif

namespace Softshell
{
#ifdef __CUDACC__
  namespace RoundRobinAlloc
  {
    extern __constant__ uint wpUseRoundRobinAlloc;
    __device__ void* alloc(uint req_size);
  }
#endif

  class Scheduler;
  class Workpackage
  {
  friend class CombWorkpackageBase;
    friend class Aggregation;
    uint eventId;
    uint eventLaunchId;
    const uint comboWorkpackage;
  protected:
    __device__ Workpackage(DEvent const & event, uint state) : eventId(event.eventId), eventLaunchId(event.eventLaunchId), comboWorkpackage(state==1?0:state==2?0x40000000:0x80000000)
    {
    }
  public:
    __device__ Workpackage(DEvent const & event) : eventId(event.eventId), eventLaunchId(event.eventLaunchId), comboWorkpackage(0x80000000)
    {
    }
 #ifdef __CUDACC__
    __device__ uint getEventId() const volatile
    {
      return eventId;
    }
    __device__ uint getEventLaunchId() const volatile
    {
      return eventLaunchId;
    }
    __device__ DEvent getEvent() const volatile
    {
      return DEvent(eventId, eventLaunchId);
    };
    __device__ bool isCombo() const volatile
    {
      return !(comboWorkpackage & 0x80000000);
    }
    __device__ bool isPayloaded()
    {
      return (comboWorkpackage & 0xC0000000) == 0x40000000;
    }

    __device__ void* operator new(size_t size)
    {
      return RoundRobinAlloc::wpCheckedAlloc(size);
    }
    __device__ void operator delete(void* p)
    {
      if(!RoundRobinAlloc::wpUseRoundRobinAlloc)
        free(p);
    }

    //cuda compiler also wants us to override placement new..
     __device__ void* operator new(size_t size, void *ptr)
    {
      return ptr;
    }
    __device__ void operator delete(void* p, void* ptr)
    {
    }

#endif
    typedef void ParamsType;
  };
  class WorkpackageEquivalent
  {
  public:
    uint eventId;
    uint eventLaunchId;
    uint comboWorkpackage;
  };

  class BlockWorkpackage : public Workpackage
  {
    public:

      uint3 blockId;
      uint numThreads;
      __device__ BlockWorkpackage(DEvent const & devent, uint3 _blockId, uint _numThreads) : Workpackage(devent), blockId(_blockId), numThreads(_numThreads)
      {
      }

  };
  template<class TParamsType>
  class BlockWorkpackageParams : public BlockWorkpackage
  {
    public:
      typedef TParamsType ParamsType;
      ParamsType* param;
      __device__ BlockWorkpackageParams(DEvent const & devent, uint3 _blockId, uint _numThreads, ParamsType& _param) : BlockWorkpackage(devent, _blockId, _numThreads), param(&_param)
      {
      }

  };
};


#endif // SOFTSHELL_API_WORKPACKAGE_INCLUDED
