/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/


#ifndef SOFTSHELL_COMMUNICATOR_DCOMMUNICATOR_IPP_INCLUDED
#define SOFTSHELL_COMMUNICATOR_DCOMMUNICATOR_IPP_INCLUDED

#include "communicator/dcommunicator.cuh"
#include "tools/utils.h"
#include <cstdio>

namespace Softshell
{
    template<class T>
    inline __device__ void callReceiver(void*receiver, MessageHeader& h, volatile uint* data, uint linid, uint threads)
    {
      reinterpret_cast<T*>(receiver)->receiveMessage(h, data, linid, threads);
    }

    __device__ DCommunicator::DCommunicator(MessageBuffer* in, MessageBuffer* out)
    {
        inbuffer = in, outbuffer = out;

        in->pos = 0;
        in->filllevel = 0;
        in->written = 0;

        out->pos = 0;
        out->filllevel = 0;
        out->written = 0;

        inbuffer_lastwritten = 0;
        already_read = 0;

        for(uint i = 0; i < ReceiverEntries; ++i)
          receivers[i].receiver = 0, receivers[i].func = 0;

        registerMessageReceiver(this, H2D_ElementsReadFromBuffer);
        registerMessageReceiver(this, H2D_Dummy);
        registerMessageReceiver(this, H2D_PING);
        registerMessageReceiver(this, H2D_PONG);
    }
    template<class T>
    __device__ bool DCommunicator::registerMessageReceiver(T* receiver, uint id)
    {

      //printf("registering message receiver for id %d: %llx %llx\n",id,receiver, &callReceiver<T>);
      if(id >= ReceiverEntries)
        printf("DCommunicator Error: Message Id (%d) too high. Maximum supported Id (ReceiverEntries) = %d\n",id,ReceiverEntries);

      //if(id >= receivers.size())
      //{
      //  receivers.resize(id+1, ReceiverEntry(0,0));
      //}


      if(receivers[id].receiver != 0)
      {
        #if GPUSCHEDULING_DEBUG_MODE >= 1
        printf("DCommunicator: error Message Receiver for ID %d already registerted: %llx\n",id,receivers[id].receiver);
        #endif
        return false;
      }
      receivers[id].receiver = reinterpret_cast<void*>(receiver);
      receivers[id].func = &callReceiver<T>;
      //printf("registered message receiver for id %d: %llx %llx\n",id,receivers[id].receiver,receivers[id].func );
      return true;
    }
    __device__ bool DCommunicator::send(uint id, uint size, uint* message)
    {
  #if GPUSCHEDULING_DEBUG_MODE >= 2
      printf("device: sending message with id %d and size %d: ",id,size);
      for(uint i = 0; i < size/sizeof(uint); ++i)
        printf(" %d ", message[i]);
      printf("\n");
  #endif
      //take care of padding
      uint paddedsize = (size + (MessageHeader::MessagePaddingBytes-1))&(~(MessageHeader::MessagePaddingBytes-1));
      uint packageSize = (sizeof(MessageHeader) + paddedsize + sizeof(uint) - 1)/sizeof(uint);
      uint old = atomicAdd(&outbuffer->filllevel, packageSize);
      if(old + packageSize <= MessageBuffer::BUFFER_SIZE)
      {
        uint spot = atomicAdd(&outbuffer->pos, packageSize) % MessageBuffer::BUFFER_SIZE;
        MessageHeader h(id,size);
        uint hsize = sizeof(h)/sizeof(uint);
        for(uint i = 0, *p = (uint*)&h; i < hsize; ++i)
        {
          outbuffer->buffer[spot] = *p++;
          spot = (spot+1)%MessageBuffer::BUFFER_SIZE;
        }
        for(uint i = 0, *p = message; i < size; i+=sizeof(uint))
        {
          outbuffer->buffer[spot] = *p++;
          spot = (spot+1)%MessageBuffer::BUFFER_SIZE;
        }
        __threadfence_system();
        atomicAdd((uint*)&outbuffer->written, packageSize);
        return true;
      }
      else
      {
        atomicSub(&outbuffer->filllevel, packageSize);
        return false;
      }
    }
    __device__ int DCommunicator::dispatch(uint linid, uint threads)
    {
      __shared__ MessageHeader h;
      __shared__ volatile uint* data;
      __shared__ volatile bool run;
      uint current_lastwritten = inbuffer->written;
      if(linid == 0)
        run = inbuffer_lastwritten != current_lastwritten;
      __syncthreads();
      if(run)
      {
        uint read = 0;
        uint spot = inbuffer_lastwritten % MessageBuffer::BUFFER_SIZE;
        uint paddedsize;
        __syncthreads();
        while(run)
        {
          if(linid == 0)
          {

            uint hsize = sizeof(h)/sizeof(uint);
            for(uint i = 0, *p = (uint*)&h; i < hsize; ++i)
            {
              *p++ = inbuffer->buffer[spot];
              spot = (spot+1)%MessageBuffer::BUFFER_SIZE;
            }

            paddedsize = (h.size + (MessageHeader::MessagePaddingBytes-1))&(~(MessageHeader::MessagePaddingBytes-1));
            paddedsize = (paddedsize + sizeof(uint) - 1)/sizeof(uint);
            data = inbuffer->buffer + spot;
            //that can not happen anymore!
            //if(spot + paddedsize > MessageBuffer::BUFFER_SIZE)
            //{
            //  data = new uint[paddedsize];
            //  for(uint i = 0, *p = data, s = spot; i < paddedsize; ++i)
            //  {
            //    *p++ = inbuffer->buffer[s];
            //    s = (s+1)%MessageBuffer::BUFFER_SIZE;
            //  }
            //}
  #if GPUSCHEDULING_DEBUG_MODE >= 2
            printf("device: received message with id %d @ pos %d, spots %d: \n",h.id,inbuffer_lastwritten,paddedsize);
            //for(uint i = 0; i < paddedsize; ++i)
            //  printf(" %d", data[i]);
            //printf("\n");
            //for(uint i = 0; i < paddedsize; ++i)
            //  printf(" %d", inbuffer->buffer[spot+i]);
            //printf("\n");
  #endif

            inbuffer_lastwritten = inbuffer_lastwritten + paddedsize + hsize;
            read = read + paddedsize + hsize;
            if(inbuffer_lastwritten == current_lastwritten)
              run = false;

          }
          __syncthreads();

          //if(h.id < receivers.size())
          if(h.id < ReceiverEntries)
          {
            if(receivers[h.id].receiver == 0)
              this->receiveMessage(h, data, linid, threads);
            //call the receiver
            //if(linid == 0)
            //  printf("calling receiver: %llx %llx\n",receivers[h.id].receiver,receivers[h.id].func);
            //
            receivers[h.id].func(receivers[h.id].receiver, h, data, linid, threads);


            //receivers[h.id]->receiveMessage(h, data, linid, threads);
          }
          else
            this->receiveMessage(h, data, linid, threads);
          __syncthreads();

          if(linid == 0)
          {
            //if(spot + paddedsize > MessageBuffer::BUFFER_SIZE)
            //  delete[] data;
            spot = (spot + paddedsize) % MessageBuffer::BUFFER_SIZE;
          }
        }
        if(linid == 0 && read != 0)
        {
          read = already_read + read;
          //avoid ping pong with ElementsReadFromBuffer Messages
          if(read > ReportReadThreshold)
          {
            send(D2H_ElementsReadFromBuffer,sizeof(uint),&read);
            read = 0;
          }
          already_read = read;
        }
        return 1;
      }
      return 0;
    }

    __device__ void DCommunicator::receiveMessage(MessageHeader& h, volatile uint* data, uint linid, uint threads)
    {
      if(linid ==0)
      {
      if(h.id == H2D_ElementsReadFromBuffer)
        atomicSub(&outbuffer->filllevel,*data);
      else if(h.id == H2D_Dummy || h.id == H2D_PONG)
      {
        //nothing :)
      }
      else if(h.id == H2D_PING)
      {
        send(D2H_PONG,0,0);
      }
      else
      {
        uint e[3] = {0xEEEEEE,h.id,data-inbuffer->buffer};
        send(D2H_DEBUG, 3*sizeof(uint), e);
        printf("COMMUNICATION ERROR: DEVICE got unhandled message with id: %d\n",h.id);
      }
      }
    }
 }
#endif
