/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/


#include "communicator/hcommunicator.h"
#include <iostream>
#include "tools/utils.h"

namespace Softshell
{
  void initDeviceCommunicator(MessageBuffer* h2d_queue, MessageBuffer* d2h_queue);

  HCommunicator::HCommunicator(Device* _device) : run(true), sleep(true), device(_device)
  {
    registerMessageReceiver(this);
    init();
    Thread::start();

    startsignal.wait();
  }
  HCommunicator::~HCommunicator()
  {
    kill();
  }
  void HCommunicator::init()
  {
    size_t required_memory = sizeof(MessageBuffer);
    CUDA_CHECKED_CALL(cudaHostAlloc(&h_HostToDeviceBuffer, required_memory, cudaHostAllocMapped));
    CUDA_CHECKED_CALL(cudaHostAlloc(&h_DeviceToHostBuffer, required_memory, cudaHostAllocMapped));
    CUDA_CHECKED_CALL(cudaHostGetDevicePointer(&d_HostToDeviceBuffer, h_HostToDeviceBuffer, 0));
    CUDA_CHECKED_CALL(cudaHostGetDevicePointer(&d_DeviceToHostBuffer, h_DeviceToHostBuffer, 0));

    //CUDA_CHECKED_CALL(cudaMalloc((void**)&d_HostToDeviceBuffer, sizeof(MessageBuffer)));
    //CUDA_CHECKED_CALL(cudaMalloc((void**)&d_DeviceToHostBuffer, sizeof(MessageBuffer)));

    inbuffer_lastwritten = 0;
    already_read = 0;
    initDeviceCommunicator(d_HostToDeviceBuffer, d_DeviceToHostBuffer);
  }
  int HCommunicator::execute()
  {
    startsignal.signal();
    bool should_yield = false;
    while(run)
    {
      if(sleep) signal.wait();
      if(run && !sleep)
      {
        //process new messages
        if(processMessages() == 0)
        {
          //wait a little till we check again
          if(should_yield)
            yield();
          else
          {
            should_yield = true;
            for(int i = 0; i < 1000; ++i);
          }
        }
        else
          should_yield = false;
      }
    }
    cleanup();
    return 0;
  }
  void HCommunicator::cleanup()
  {
    CUDA_CHECKED_CALL(cudaFreeHost(h_HostToDeviceBuffer));
  }
  uint HCommunicator::processMessages()
  {
    uint count = 0;
    cudaStreamQuery(0);

    if(h_DeviceToHostBuffer->written != inbuffer_lastwritten)
    {
      count = h_DeviceToHostBuffer->written - inbuffer_lastwritten;

      bool run = true;
      MessageHeader h;
      uint* data;
      uint read = 0;
      uint spot = inbuffer_lastwritten % MessageBuffer::BUFFER_SIZE;
      uint paddedsize;
      while(run)
      {
        uint hsize = sizeof(h)/sizeof(uint);
        for(uint i = 0, *p = (uint*)&h; i < hsize; ++i)
        {
          *p++ = h_DeviceToHostBuffer->buffer[spot];
          spot = (++spot)%MessageBuffer::BUFFER_SIZE;
        }
        paddedsize = (h.size + (MessageHeader::MessagePaddingBytes-1))&(~(MessageHeader::MessagePaddingBytes-1));
        paddedsize = (paddedsize + sizeof(uint) - 1)/sizeof(uint);

        data = (uint*) h_DeviceToHostBuffer->buffer + spot;
        if(spot + paddedsize > MessageBuffer::BUFFER_SIZE)
        {
          data = new uint[paddedsize];
          for(uint i = 0, *p = data, s = spot; i < paddedsize; ++i)
          {
            *p++ = h_DeviceToHostBuffer->buffer[s];
            s = (++s)%MessageBuffer::BUFFER_SIZE;
          }
        }
        inbuffer_lastwritten += paddedsize + hsize;
        read += paddedsize + hsize;
        if(inbuffer_lastwritten == h_DeviceToHostBuffer->written)
          run = false;

        //printf("received message: %d %d\n",h.id, h.size);

        if(receivers[h.id])
          receivers[h.id]->receiveMessage(h, data, this, device);
        else
          std::cerr << "COMMUNICATION ERROR HOST received unknown message with id " << h.id << std::endl;

        if(spot + paddedsize > MessageBuffer::BUFFER_SIZE)
          delete[] data;
        spot = (spot + paddedsize) % MessageBuffer::BUFFER_SIZE;
      }
      already_read += read;
      //avoid ping pong with ElementsReadFromBuffer Messages
      if(already_read > ReportReadThreshold)
      {
        send(H2D_ElementsReadFromBuffer,sizeof(uint),&already_read);
        already_read = 0;
      }
    }
    return count;
  }

  void HCommunicator::registerMessageReceiver(HMessageReceiver* receiver)
  {
    uint num = receiver->numReceivingMessages();
    for(uint i = 0; i < num; ++i)
    {
      uint id = receiver->receivingMessageId(i);
      if(id >= receivers.size())
        receivers.resize(id+1,0);
      receivers[id] = receiver;
    }
  }
  bool HCommunicator::send(uint messageId, uint size, uint* message)
  {
    Guard guard(sendbufferMutex);
    //padding
    uint paddedsize = (size + (MessageHeader::MessagePaddingBytes-1))&(~(MessageHeader::MessagePaddingBytes-1));
    uint packageSize = (sizeof(MessageHeader) + paddedsize + sizeof(uint) - 1)/sizeof(uint);
    //std::cout << "sending message " << getH2DMessageName(messageId) << std::endl;

    uint hsize = sizeof(MessageHeader)/sizeof(uint);
    //note that it is ok for the message header to span
    if(h_HostToDeviceBuffer->pos + packageSize >= MessageBuffer::BUFFER_SIZE && h_HostToDeviceBuffer->pos + hsize < MessageBuffer::BUFFER_SIZE)
    {
      //we need a dummy message
      uint dummysize = MessageBuffer::BUFFER_SIZE - h_HostToDeviceBuffer->pos;
      if(h_HostToDeviceBuffer->filllevel + packageSize + dummysize >= MessageBuffer::BUFFER_SIZE)
        return false;
      //send the dummy, note that it is ok for the message header to span
      uint spot = h_HostToDeviceBuffer->pos;
      MessageHeader h(H2D_Dummy,(dummysize-hsize)*sizeof(uint));
      for(uint i = 0, *p = (uint*)&h; i < hsize; ++i)
      {
        h_HostToDeviceBuffer->buffer[spot] = *p++;
        spot = (++spot)%MessageBuffer::BUFFER_SIZE;
      }
      h_HostToDeviceBuffer->pos = (h_HostToDeviceBuffer->pos + dummysize) % MessageBuffer::BUFFER_SIZE;
      h_HostToDeviceBuffer->filllevel += dummysize;
      h_HostToDeviceBuffer->written += dummysize;
    }
    if(h_HostToDeviceBuffer->filllevel + packageSize < MessageBuffer::BUFFER_SIZE)
    {
      uint spot = h_HostToDeviceBuffer->pos;
      MessageHeader h(messageId,size);
      for(uint i = 0, *p = (uint*)&h; i < hsize; ++i)
      {
        h_HostToDeviceBuffer->buffer[spot] = *p++;
        spot = (++spot)%MessageBuffer::BUFFER_SIZE;
      }
      for(uint i = 0, *p = message; i < size; i+=sizeof(uint))
      {
        h_HostToDeviceBuffer->buffer[spot] = *p++;
        spot = (++spot)%MessageBuffer::BUFFER_SIZE;
      }

      h_HostToDeviceBuffer->pos = (h_HostToDeviceBuffer->pos + packageSize) % MessageBuffer::BUFFER_SIZE;
      h_HostToDeviceBuffer->filllevel += packageSize;
      h_HostToDeviceBuffer->written += packageSize;
      return true;
    }
    else
      return false;
  }
  void HCommunicator::start()
  {
    sleep = false;
    signal.signal();
  }
  void HCommunicator::pause()
  {
    sleep = true;
  }
  void HCommunicator::shutdown()
  {
    run = false;
    sleep = false;
    signal.signal();
  }
  void HCommunicator::join()
  {
    Thread::join();
  }

  uint HCommunicator::numReceivingMessages() const
  {
    return 2;
  }
  uint HCommunicator::receivingMessageId(uint i) const
  {
    switch(i)
    {
    case 0:
      return D2H_ElementsReadFromBuffer;
    case 1:
      return D2H_DEBUG;
    default:
      return 0;
    }
  }
  void HCommunicator::receiveMessage(MessageHeader& h, uint* data, HCommunicator* communicator, Device* source)
  {
    if(h.id == D2H_ElementsReadFromBuffer)
    {
      Guard guard(sendbufferMutex);
      h_HostToDeviceBuffer->filllevel -= *data;
    }
    else if(h.id == D2H_DEBUG)
    {
      printf("received debug message: ");
      for(uint i = 0; i < h.size/sizeof(uint); ++i)
       printf("%x, ", data[i]);
      printf("\n");

    }
  }
}
