/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/


#ifndef SOFTSHELL_COMMUNICATOR_HPARAMS_H_INCLUDED
#define SOFTSHELL_COMMUNICATOR_HPARAMS_H_INCLUDED

#include "communicator/hcommunicator.h"
#include "communicator/messageids.h"
#include "communicator/params.h"
#include "tools/config.h"
#include "tools/thread.h"
#include "tools/any.h"

#include <map>

namespace Softshell
{
  class HParams : public HMessageReceiver
  {
  public:
    class ArgumentHandlerBase
    {
    public:
      virtual void handle(Config* config, const std::string& name, uint* data) = 0;
      virtual ArgumentHandlerBase* clone() = 0;
      virtual ~ArgumentHandlerBase() { }
    };
    template<class T>
    class ArgumentHandlerImpl : public ArgumentHandlerBase
    {
    public:
      virtual void handle(Config* config, const std::string& name, uint* data)
      {
         Param<T>* p = reinterpret_cast<Param<T>*>(data);
         config->set(name, p->data);
      }
      ArgumentHandlerBase* clone()
      {
        return new ArgumentHandlerImpl(*this);
      }
    };
    //class ArgumentHandler
    //{
    //public:
    //  ArgumentHandlerBase* handler;
    //  ArgumentHandler() : handler(0) { }
    //  template<class T>
    //  ArgumentHandler(T dummy) : handler( new ArgumentHandlerImpl<T>() ) { }
    //  ArgumentHandler(const ArgumentHandler& other) : handler(other.handler->clone()) { }
    //  const ArgumentHandler& operator = (const ArgumentHandler& other) { handler = other.handler->clone(); }
    //  ~ArgumentHandler() { if(handler) delete handler; }
    //};
    typedef Any<ArgumentHandlerBase> ArgumentHandler;

  private:
    Config* _config;
    HCommunicator* _communicator;

    ParamId currentIgnore;
    Mutex mutex;

    typedef std::map<ParamId, ArgumentHandler> ParamHandlersType;
    ParamHandlersType paramHandlers;

  public:

    HParams(HCommunicator* communicator, Config* config) : _communicator(communicator), _config(config), currentIgnore(P_Nothing)
    {
      communicator->registerMessageReceiver(this);
    }

    template<class T>
    T addParam(ParamId id, T defaultValue)
    {
      const char* name = toString(id);
      if(name == 0) return defaultValue;
      T val = _config->registerConfig(name, defaultValue, changedParamCallback<T>, this);
      paramHandlers.insert(std::make_pair(id, ArgumentHandler(new ArgumentHandlerImpl<T>())));
      return val;
    }

    template<class T>
    T* getParam(ParamId id)
    {
      const char* name = toString(id);
      if(name == 0) return 0;
      try
      {
        return &_config->get<T>(name);
      }
      catch (std::runtime_error& err)
      {
        return 0;
      }
    }

    template<class T>
    const T* getParam(ParamId id) const
    {
      const char* name = toString(id);
      if(name == 0) return 0;
      try
      {
        return &_config->get<T>(name);
      }
      catch (std::runtime_error& err)
      {
        return 0;
      }
    }

    uint numReceivingMessages() const { return 1;}
    uint receivingMessageId(uint i) const { return D2H_Param; }
    void receiveMessage(MessageHeader& h, uint* data, HCommunicator* communicator, Device* source)
    {
      if(h.id == D2H_Param)
      {
        ParamBase *pb = reinterpret_cast<ParamBase*>(data);
        ParamHandlersType::iterator found = paramHandlers.find(pb->paramId);
        if(found != paramHandlers.end())
        {
          std::string name = toString(pb->paramId);
          Guard g(mutex);
          currentIgnore = pb->paramId;
          found->second->handle(_config, name, data);
        }
      }
    }

    Config* getConfig() { return _config; }

    template<class T>
    static void changedParamCallback(const std::string& name, T& val, HParams* me)
    {
      ParamId paramId = toParamId(name);
      if(me->currentIgnore == paramId || paramId == P_Nothing)
        return;

      Param<T> p;
      p.paramId = paramId;
      p.data = val;
      me->_communicator->send(H2D_Param, sizeof(Param<T>), reinterpret_cast<uint*>(&p));
    }
  };


}


#endif //SOFTSHELL_COMMUNICATOR_HPARAMS_INCLUDED
