/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#include "device.h"

#include "communicator/hcommunicator.h"
#include "timing/timesync.h"
#include "megakernel/megakernelissuer.h"
#include "megakernel/megakernel.h"
#include "cuda_runtime_api.h"
#include "tools/utils.h"
#include "memory/mappedMemory.h"
#include "communicator/hparams.h"
#include <iostream>
#include <sstream>
#include <string>

namespace Softshell
{
    template<class N>
    std::string ntos(N number)
    {
      std::stringstream sstr;
      sstr << number;
      return sstr.str();
    }
    Device::Device(uint deviceId, const Flags& flags, ConfigCollection* _config, const ControllerInfo * info) : _deviceId(deviceId), activeEvents(0), running(false), config(*_config, ntos(deviceId))
    {
      cudaDeviceProp prop;
      CUDA_CHECKED_CALL(cudaGetDeviceProperties(&prop, deviceId));
      name = std::string(prop.name);
#ifdef _DEBUG
      std::cout << "Softshell: Going to use " << name << std::endl;
#endif
      CUDA_CHECKED_CALL(cudaSetDeviceFlags(cudaDeviceMapHost));
      CUDA_CHECKED_CALL(cudaSetDevice(_deviceId));

      prepareDevice(flags);
      mappedMemory = new MappedMemory();
      communicator = new HCommunicator(this);
      hparams = new HParams(communicator, &config);
      if(info)
        issuer = new MegaKernelIssuer(&config, communicator, _deviceId, *info);
      else
        issuer = new MegaKernelIssuer(&config, communicator, _deviceId);
      timesync = new TimeSync(this, &config);
      initMegaKernel(flags, hparams);
    }
    Device::~Device()
    {
      shutdown();
    }
    float Device::calcScore(EventBase* event)
    {
      //TODO put the intelligence in here tochoose the best device
      return 1.0f/static_cast<float>(activeEvents);
    }

    uint Device::deviceId() const
    {
      return _deviceId;
    }

    HCommunicator* Device::getCommunicator()
    {
      return communicator;
    }
    MappedMemory* Device::getMappedMemory()
    {
      return mappedMemory;
    }

    const std::string& Device::getName()
    {
      return name;
    }

    void  Device::execEvent(uint eventId, uint eventLaunchId)
    {
      Guard g(eventMutex);
      EventCallCountType::iterator found = eventCallCount.find(eventId);
      if(found == eventCallCount.end())
        found = eventCallCount.insert(std::make_pair(eventId,EventExecInfo())).first;

      ++found->second.activeInstances;
      ++found->second.callCount;
      ++activeEvents;

      start();
    }
    void  Device::completedEvent(uint eventId, uint eventLaunchId, float time)
    {
      Guard g(eventMutex);
      EventCallCountType::iterator found = eventCallCount.find(eventId);
      if(found != eventCallCount.end())
      {
        --found->second.activeInstances;
        if(--activeEvents <= 0)
        {
          joinedAll.signal();
          activeEvents = 0;
          pause();
        }
      }
    }

    void Device::pause()
    {
      Guard g(stateMutex);
      running = false;
      timesync->pause();
      issuer->pause();
      issuer->stopActiveLaunches();
      communicator->pause();
    }
    void Device::start()
    {
      Guard g(stateMutex);
      if(running)
        return;
      if(activeEvents <= 0)
        return;
      running = true;
      communicator->start();
      issuer->start();
      timesync->start();

    }
    void Device::shutdown()
    {
      if(communicator == 0 || issuer == 0 || timesync == 0)
        return;
      issuer->end();
      //printf("before issuer join\n");
      issuer->join();
      //printf("after issuer join\n");
      communicator->shutdown();
      communicator->join();
      timesync->shutdown();
      timesync->join();
      delete communicator;
      delete issuer;
      delete timesync;
      communicator = 0;
      issuer = 0;
      timesync = 0;
    }

    void Device::join()
    {
      Guard g(eventMutex);
      if(activeEvents == 0)
        return;
      g.unlock();
      joinedAll.wait();
    }
}
