/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*   Rostislav Khlebnikov /  khlebnikov ( at ) icg.tugraz.at
*/

#ifndef SOFTSHELL_DISTRIBUTOR_DEVICEENTRIES_CUH_INCLUDED
#define SOFTSHELL_DISTRIBUTOR_DEVICEENTRIES_CUH_INCLUDED

#include "distributor/deviceentries.h"
#include "distributor/defaultcreatorparams.h"
#include "tools/utils.h"
#include <stdio.h>

#ifdef __CUDACC__

#include "queue/dscheduler.cuh"
#include "distributor/eventmanager.cuh"

#ifndef USE_CUDA_HEAP
#include "memory/UseScatterAlloc.cuh"
#endif

//placement new required
//void* operator new(size_t, void*);


namespace Softshell
{
    class DEvent;

    template<class TFunctionProto, class TWrappedParam, template<class> class TDeviceWrapper>
    __global__ void getDeviceFunctionPointer(TFunctionProto* p)
    {
        *p = &TDeviceWrapper<TWrappedParam>::wrapped;
    }
    template<class TFunctionProto, class TWrappedParam, template<class> class TDeviceWrapper>
    TFunctionProto readDeviceFunctionPointer()
    {
        TFunctionProto h_res, *d_res;
        CUDA_CHECKED_CALL(cudaMalloc(&d_res, sizeof(TFunctionProto)));
        getDeviceFunctionPointer<TFunctionProto, TWrappedParam, TDeviceWrapper><<<1,1>>>(d_res);
        CUDA_CHECKED_CALL(cudaMemcpy(&h_res, d_res, sizeof(TFunctionProto), cudaMemcpyDeviceToHost));
        return h_res;
    }

    template<class TWorkpackage, class WorkpackageParamsType>
    __device__ inline TWorkpackage* newDefaultWorkpackage(DEvent devent, const uint3& id, DefaultCreatorParams<WorkpackageParamsType>* creatorParams)
    {
      return new TWorkpackage(devent, id, creatorParams->threads, creatorParams->workpackageParams);
    }
    template<class TWorkpackage, class WorkpackageParamsType>
    __device__ inline TWorkpackage* placementNewDefaultWorkpackage(TWorkpackage* wp, DEvent devent, const uint3& id, DefaultCreatorParams<WorkpackageParamsType>* creatorParams)
    {
      return new(wp) TWorkpackage(devent, id, creatorParams->threads, creatorParams->workpackageParams);
    }
    template<class TWorkpackage>
    __device__ inline TWorkpackage* newDefaultWorkpackage(DEvent devent, const uint3& id, DefaultCreatorParams<void>* creatorParams)
    {
      return new TWorkpackage(devent, id, creatorParams->threads);
    }
    template<class TWorkpackage>
    __device__ inline TWorkpackage* placementNewDefaultWorkpackage(TWorkpackage* wp, DEvent devent, const uint3& id, DefaultCreatorParams<void>* creatorParams)
    {
      return new(wp) TWorkpackage(devent, id, creatorParams->threads);
    }


    template<class TWorkpackage, class TProcedure, class WorkpackageParamsType>
    class DefaultEventCreator
    {
    public:
        typedef DefaultCreatorParams<WorkpackageParamsType> CreatorParamsType;

        __device__ static void create(uint eventId, uint eventLaunchId, CreatorParamsType* creatorParams, void* &storage)
        {
            uint linid = threadIdx.x;
            uint threads = blockDim.x;
            uint sumLaunch = creatorParams->workpackages_x*creatorParams->workpackages_y*creatorParams->workpackages_z;
           
            __shared__ TWorkpackage* wps;
            wps = reinterpret_cast<TWorkpackage*>(storage);
            if(threadIdx.x == 0)
            {
              //printf("will create %d wps for %d %d\n",sumLaunch,eventId,eventLaunchId);
              d_eventManager.newWorkpackagesForEvent(eventId, eventLaunchId, sumLaunch);
              d_procedureManager->ensureAvailability<TProcedure>();
            }

            __syncthreads();
            if(creatorParams->ewm == EWM_CreateOnly)
            {
              for(uint i = linid; i < sumLaunch; i+=threads)
              {
                uint3 id;
                uint t = i;
                id.z = t / (creatorParams->workpackages_x*creatorParams->workpackages_y);
                t = t - id.z*(creatorParams->workpackages_x*creatorParams->workpackages_y);
                id.y = t / creatorParams->workpackages_x;
                id.x = t - id.y*creatorParams->workpackages_x;

                TWorkpackage* wp = newDefaultWorkpackage<TWorkpackage>(DEvent(eventId, eventLaunchId), id, creatorParams);
                enqueueWorkpackage<TProcedure>(wp, creatorParams->threads);
              }
            }
            else if(creatorParams->ewm == EWM_LaunchLifetime)
            {
              if(linid == 0 )
              {
                //printf("allocating.. for %d %d\n",eventId, eventLaunchId);
                  wps = (TWorkpackage*)malloc(sizeof(TWorkpackage)*sumLaunch);
                  storage = wps;
              }
              __syncthreads();
              for(uint i = linid; i < sumLaunch; i+=threads)
              {
                  uint3 id;
                  uint t = i;
                  id.z = t / (creatorParams->workpackages_x*creatorParams->workpackages_y);
                  t = t - id.z*(creatorParams->workpackages_x*creatorParams->workpackages_y);
                  id.y = t / creatorParams->workpackages_x;
                  id.x = t - id.y*creatorParams->workpackages_x;
                  TWorkpackage* wp = placementNewDefaultWorkpackage(wps+i, DEvent(eventId, eventLaunchId), id, creatorParams);
                  //printf("enqueing workpackage %d %d %d\n", id.x,id.y,id.z);
                  enqueueWorkpackage<TProcedure>(wp, creatorParams->threads);
              }
            }
            else if(creatorParams->ewm == EWM_Recurring)
            {
              if(linid == 0 && storage == 0)
              {
                //printf("allocating.. for %d %d\n",eventId, eventLaunchId);
                wps = (TWorkpackage*)malloc(sizeof(TWorkpackage)*sumLaunch);
                storage = wps;
              }
              __syncthreads();
              for(uint i = linid; i < sumLaunch; i+=threads)
              {
                uint3 id;
                uint t = i;
                id.z = t / (creatorParams->workpackages_x*creatorParams->workpackages_y);
                t = t - id.z*(creatorParams->workpackages_x*creatorParams->workpackages_y);
                id.y = t / creatorParams->workpackages_x;
                id.x = t - id.y*creatorParams->workpackages_x;
                TWorkpackage* wp = placementNewDefaultWorkpackage(wps+i, DEvent(eventId, eventLaunchId), id, creatorParams);
                //printf("enqueing workpackage %d %d %d\n", id.x,id.y,id.z);
                enqueueWorkpackage<TProcedure>(wp, creatorParams->threads);
              }
            }
        }
        __device__ static void destroy(uint eventId, uint eventLaunchId, CreatorParamsType* creatorParams, void* &storage)
        {
            if(creatorParams->ewm == EWM_LaunchLifetime && threadIdx.x == 0)
            {
                free(storage);
                storage = 0;
            }
            __syncthreads();
        }
    };
}

#endif

#endif //SOFTSHELL_DISTRIBUTOR_DEVICEENTRIES_INCLUDED
