/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*   Rostislav Khlebnikov /  khlebnikov ( at ) icg.tugraz.at
*/

#ifndef __EVENT_IMPL_IPP_INCLUDED__
#define __EVENT_IMPL_IPP_INCLUDED__

//#include "distributor/event_impl.h"
#include "distributor/device.h"
#include "hscheduler.h"
#include "distributor/eventlist.h"
#include "memory/mappedMemory.h"
#include "distributor/eventData.h"

#include <limits>
//#include <iostream>

namespace Softshell
{
    template<typename ParamsType>
    std::pair<void*, void*> allocAndCopy(ParamsType* params, Device* targetDevice)
    {
      std::pair<void*,void*> ptr = targetDevice->getMappedMemory()->alloc(sizeof(ParamsType));
      memcpy(ptr.first, params, sizeof(ParamsType));
      return ptr;
    }

    inline std::pair<void*, void*> allocAndCopy(void* params, Device* targetDevice)
    {
        return std::make_pair<void*,void*>(nullptr, nullptr);
    }

    template<class TEventCreatorDestroyer>
    uint EventImpl<TEventCreatorDestroyer>::fired(const CreatorParamsType& creatorParams,
        DeviceEntryCreatorDestroyerFunc eventCreator,
        DeviceEntryCreatorDestroyerFunc eventDestroyer,
        uint deviceId)
    {
      Device* targetDevice = hscheduler->getDevice(deviceId);

      //get id
      uint launchid = EventList::getInstance()->requestLaunchId(owner);
      //printf("fireing %d %d\n",id,launchid);

      //copy params
      std::pair<void*, void*> creatorParamsPointers = allocAndCopy(&creatorParams, targetDevice);

      //inform device
      targetDevice->execEvent(id, launchid);

      //send firing message
      EventData edata;
      edata.creator = eventCreator;
      edata.destroyer = eventDestroyer;
      edata.eventId = id;
      edata.creatorParams = creatorParamsPointers.second;
      edata.eventLaunchId = launchid;

      targetDevice->getCommunicator()->send(H2D_FireEvent, sizeof(EventData), reinterpret_cast<uint*>(&edata));

      {
        Guard g(joinerMutex);
        activeLaunches.insert(std::make_pair(launchid, LaunchData(creatorParamsPointers)));
      }

      return launchid;
    }
    template<class TEventCreatorDestroyer>
    uint EventImpl<TEventCreatorDestroyer>::chooseDevice(Device* device)
    {
      if(device != 0)
        return device->deviceId();
      else
      {
        float score = -std::numeric_limits<float>::max();
        //iterate over devices and return appropriate
        for(HScheduler::DeviceToIDMappingType::const_iterator it = hscheduler->deviceToIdMapping.begin(); it != hscheduler->deviceToIdMapping.end(); ++it)
        {
          float tscore = it->first->calcScore(owner);
          if(tscore > score)
            score = tscore,
            device = it->first;
        }
        return device->deviceId();
      }
    }

    template<class TEventCreatorDestroyer>
    EventImpl<TEventCreatorDestroyer>::EventImpl(HScheduler* _hscheduler, Event<TEventCreatorDestroyer>* owner) :
      hscheduler(_hscheduler),
      owner(owner),
      id(EventList::getInstance()->requestId(owner))
    {

    }
    template<class TEventCreatorDestroyer>
    EventImpl<TEventCreatorDestroyer>::~EventImpl()
    {
      EventList::getInstance()->destroyed(owner);
    }

    template<class TEventCreatorDestroyer>
    void EventImpl<TEventCreatorDestroyer>::registerNotifier(EventNotifier* notifier)
    {
      Guard g(notifierMutex);
      notifiers.insert(notifier);
    }
    template<class TEventCreatorDestroyer>
    void EventImpl<TEventCreatorDestroyer>::unregisterNotifier(EventNotifier* notifier)
    {
      Guard g(notifierMutex);
      notifiers.erase(notifier);
    }

    template<class TEventCreatorDestroyer>
    void EventImpl<TEventCreatorDestroyer>::join()
    {
      Guard g(joinerMutex);
      if(activeLaunches.size() == 0)
        return;
      g.unlock();
      joinedAll.wait();
    }
    template<class TEventCreatorDestroyer>
    void EventImpl<TEventCreatorDestroyer>::join(uint id)
    {
      Guard g(joinerMutex);
      if(activeLaunches.size() == 0)
        return;

      //FIXME: the following line does not work for some strange reason, requiring multiple find calls -> probably inefficient

      //ActiveLaunchesType::const_iterator found = activeLaunches.find(id);

      //ActiveLaunchesType::it;
      //    =
      //    activeLaunches.find(id);
      //std::map<uint, LaunchData > found = activeLaunches.find(id);
      //bool found = true;

      if(activeLaunches.find(id) != activeLaunches.end())
      {
        Signal& sig(activeLaunches.find(id)->second.signal);
        g.unlock();
        sig.wait();
      }
    }

    template<class TEventCreatorDestroyer>
    void EventImpl<TEventCreatorDestroyer>::cancel()
    {
      //TODO
    }
    template<class TEventCreatorDestroyer>
    void EventImpl<TEventCreatorDestroyer>::cancel(uint id)
    {
      //TODO
    }

    template<class TEventCreatorDestroyer>
    uint EventImpl<TEventCreatorDestroyer>::getNumActiveLaunches() const
    {
      Guard g(const_cast<Mutex&>(joinerMutex));
      return activeLaunches.size();
    }
    template<class TEventCreatorDestroyer>
    std::vector<EventStatistics> EventImpl<TEventCreatorDestroyer>::getEventStatus() const
    {
      return std::vector<EventStatistics>();
    }

    template<class TEventCreatorDestroyer>
    void EventImpl<TEventCreatorDestroyer>::setPeriodicity(float periodicity)
    {
      uint2 data;
      data.x = id;
      data.y = *reinterpret_cast<uint*>(&periodicity);
      for(HScheduler::IDToDeviceMappingType::const_iterator it = hscheduler->idToDeviceMapping.begin(); it != hscheduler->idToDeviceMapping.end(); ++it)
        it->second->getCommunicator()->send(H2D_EventPeriodicity,2*sizeof(uint),(uint*)&data);
    }
    template<class TEventCreatorDestroyer>
    void EventImpl<TEventCreatorDestroyer>::setDeadline(float deadline)
    {
      uint2 data;
      data.x = id;
      data.y = *reinterpret_cast<uint*>(&deadline);
      for(HScheduler::IDToDeviceMappingType::const_iterator it = hscheduler->idToDeviceMapping.begin(); it != hscheduler->idToDeviceMapping.end(); ++it)
        it->second->getCommunicator()->send(H2D_EventDeadline,2*sizeof(uint),(uint*)&data);
    }
    template<class TEventCreatorDestroyer>
    void EventImpl<TEventCreatorDestroyer>::setPriority(Priority priority)
    {
      uint2 data;
      data.x = id;
      data.y = priority;
      for(HScheduler::IDToDeviceMappingType::const_iterator it = hscheduler->idToDeviceMapping.begin(); it != hscheduler->idToDeviceMapping.end(); ++it)
        it->second->getCommunicator()->send(H2D_EventPriority,2*sizeof(uint),(uint*)&data);
    }


    template<class TEventCreatorDestroyer>
    void EventImpl<TEventCreatorDestroyer>::done(uint launchId, Device* device, float time)
    {
      //TODO update statistics
      std::pair<void*,void*> memptr = std::make_pair<void*,void*>(nullptr,nullptr);
      {
        Guard g(joinerMutex);
        //ActiveLaunchesType::iterator found = activeLaunches.find(launchId);
        if(activeLaunches.find(launchId) != activeLaunches.end())
        {
          execTimes.insert(std::pair<uint, double>(launchId, time));
          activeLaunches.find(launchId)->second.signal.signal();
          memptr = activeLaunches.find(launchId)->second.allocatedMemory;
          activeLaunches.erase(activeLaunches.find(launchId));
          if(activeLaunches.size() == 0)
            joinedAll.signal();
        }
      }

      {
        Guard g(notifierMutex);
        for(NotifiersType::const_iterator it = notifiers.begin(); it != notifiers.end(); ++it)
          (*it)->completed(owner, launchId, time, memptr.first);
      }
      device->getMappedMemory()->free(memptr);
    }

    template<class TEventCreatorDestroyer>
    double EventImpl<TEventCreatorDestroyer>::getExecTime(uint id)
    {
      Guard g(joinerMutex);
      ExecTimesType::const_iterator found = execTimes.find(id);
      if(found != execTimes.end())
        return found->second;
      else
        return -1;
    }
}

#endif
