/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*   Rostislav Khlebnikov /  khlebnikov ( at ) icg.tugraz.at
*/

#ifndef SOFTSHELL_DISTIBUTOR_EVENTMANAGER_IPP_INCLUDED
#define SOFTSHELL_DISTIBUTOR_EVENTMANAGER_IPP_INCLUDED

#include "tools/common.cuh"
#include "timing/timesync.cuh"
#include "api/devent.h"

namespace Softshell
{

  __device__ void EventManager::init()
  {
    for(uint id = threadIdx.x + blockIdx.x*blockDim.x;
        id < EventListSize;
        id += blockDim.x*gridDim.x)
      eventInfos[id].init();
    for(uint id = threadIdx.x + blockIdx.x*blockDim.x;
        id < EventLaunchListSize;
        id += blockDim.x*gridDim.x)
      eventLaunchInfos[id].init();
     //for(uint id = threadIdx.x + blockIdx.x*blockDim.x;
     //   id < 1024;
     //   id += blockDim.x*gridDim.x)
     // WorkItemCounter[id] = 0;
  }
  __device__ Priority EventManager::getPriority(uint eventId) const
  {
    return eventInfos[eventId%EventListSize].priority;
  }
  __device__ void EventManager::setPriority(uint eventId, Priority priority, bool inform)
  {
    //TODO tell CPU!
    eventInfos[eventId%EventListSize].priority = priority;
  }
  __device__ float EventManager::getDeadline(uint eventId) const
  {
    return eventInfos[eventId%EventListSize].deadline;
  }
  __device__ void EventManager::setDeadline(uint eventId, float deadline, bool inform)
  {
    //TODO tell CPU!
    eventInfos[eventId%EventListSize].deadline = deadline;
  }
  __device__ float EventManager::getPeriodicity(uint eventId) const
  {
    return eventInfos[eventId%EventListSize].periodicity;
  }
  __device__ void EventManager::setPeriodicity(uint eventId, float periodicity, bool inform)
  {
    //TODO tell CPU!
    eventInfos[eventId%EventListSize].periodicity = periodicity;
  }

  __device__  double EventManager::getLaunchTime(uint eventLaunchId) const
  {
    return eventLaunchInfos[eventLaunchId%EventLaunchListSize].starttime;
  }
  __device__  int EventManager::getActiveWorkpackages(uint eventLaunchId) const
  {
    return eventLaunchInfos[eventLaunchId%EventLaunchListSize].activeworkpackagecount;
  }
  __device__  int EventManager::getProcessedWorkpackages(uint eventLaunchId) const
  {
    return eventLaunchInfos[eventLaunchId%EventLaunchListSize].processedworkpackagecount;
  }

  __device__ void EventManager::newWorkpackagesForEvent(const DEvent& devent, int num)
  {
    int old = atomicAddVolatile(&eventLaunchInfos[devent.eventLaunchId%EventLaunchListSize].activeworkpackagecount, num);
    //printf("new wp: %d\n",old+num);
  }
  __device__ void EventManager::newWorkpackagesForEvent(uint eventId, uint eventLaunchId, int num)
  {
    int old = atomicAddVolatile(&eventLaunchInfos[eventLaunchId%EventLaunchListSize].activeworkpackagecount, num);
    //printf("new wp: %d\n",old+num);
  }
  __device__ void EventManager::finishedWorkpackagesForEvent(volatile DEvent& devent, int num)
  {
    finishedWorkpackagesForEvent(devent.eventId, devent.eventLaunchId);
  }

  __device__ void EventManager::finishedWorkpackagesForEvent(uint eventId, uint eventLaunchId, int num)
  {
    atomicAddVolatile(&eventLaunchInfos[eventLaunchId%EventLaunchListSize].processedworkpackagecount, num);
    int old = atomicSubVolatile(&eventLaunchInfos[eventLaunchId%EventLaunchListSize].activeworkpackagecount, num);
    //debug
    //if(old - num < 10)
    //if(((old - num)&0xFF)==0)
    //  printf("%d\n",old - num);


    //debug
    //atomicSub((int*)&WorkItemCounter[num],1);
    //
    //printf("num: %d\n",old - num);
    if(old - num <= 0)
    {
      __threadfence();
      //printf("ending launch %d\n",rDevent.eventLaunchId);
      volatile EventLaunchInfo& linfo(eventLaunchInfos[eventLaunchId%EventLaunchListSize]);

      //call destroyer
      //void *xptr = linfo.storage;
      linfo.destroyer(eventId, eventLaunchId, linfo.creatorParams,
                      //const_cast<void*>(linfo.storage)
                      *(void** )(&linfo.storage)
                      );
      //update event storage
      if(linfo.storage != 0)
        eventInfos[linfo.eventId%EventListSize].storage = linfo.storage;

      //inform we are done
      double t = getTime();
      double dt = t - linfo.starttime;
      struct EventReturn
      {
        uint eventId, eventLaunchId;
        float exectime;
      };
      EventReturn ret;
      ret.eventId = eventId;
      ret.eventLaunchId = eventLaunchId;
      //ret.exectime = (float)(t - linfo.starttime);
      ret.exectime = (float)cycleDiffToTime(getTimeCycles() - linfo.startcycles);
      d_communicator->send(D2H_EventLaunchDone, sizeof(EventReturn), reinterpret_cast<uint*>(&ret));

      //printf("finished call %d %d\n",ret.eventId, ret.eventLaunchId );
      //TODO update statistics

      //clear
      linfo.processedworkpackagecount = 0;

      if(old - num < 0)
        printf("WARNING workpackage count for event %d %d below 0: %d\n", eventId, eventLaunchId, old - num);
    }
  }
  __device__ void EventManager::newEventLaunch(volatile EventData* eventData)
  {
    uint eventId = eventData->eventId;
    uint eventLaunchId = eventData->eventLaunchId;
    void* creatorParams = eventData->creatorParams;
    volatile EventLaunchInfo& linfo(eventLaunchInfos[eventLaunchId%EventLaunchListSize]);
    if(threadIdx.x == 0)
    {
      double t = getTime();
      clock64_t statcylces = getTimeCycles();
      linfo.destroyer = eventData->destroyer;
      linfo.creatorParams = creatorParams;
      linfo.starttime = t;
      linfo.startcycles = statcylces;
      linfo.storage = eventInfos[eventId%EventListSize].storage;
      linfo.eventId = eventId;
    }
    __syncthreads();

    eventData->creator(eventId, eventLaunchId, creatorParams,
                       //const_cast<void*>(linfo.storage)
                       *(void**)(&linfo.storage)
                       );
    //if(threadIdx.x == 0)
    //  printf("created event for %d %d\n",devent.eventId, devent.eventLaunchId);
  }

  inline __device__ void EventManager::debugEventLaunchInfo(uint info)
  {
    for(int i = threadIdx.x; i < EventLaunchListSize; i+=blockDim.x)
    {
      if(eventLaunchInfos[i].activeworkpackagecount != 0)
      {
        uint m[4] = {0,info, i, eventLaunchInfos[i].activeworkpackagecount};
        d_communicator->send(D2H_DEBUG, 4*sizeof(uint),m);
      }
    }
  }
  //inline __device__ void EventManager::debugWorkItemCounter()
  //{
  //  __shared__ int sum;
  //  sum = 0;
  //  __syncthreads();
  //  for(int i = threadIdx.x; i < 1024; i+=blockDim.x)
  //  {
  //    if(WorkItemCounter[i] != 0)
  //    {
  //      uint m[4] = {0, 0, i, WorkItemCounter[i]};
  //      d_communicator->send(D2H_DEBUG, 4*sizeof(uint),m);
  //      atomicAdd(&sum,WorkItemCounter[i]);
  //    }
  //  }
  //  __syncthreads();
  //  if(threadIdx.x == 0)
  //  {
  //    uint m[4] = {0, 1, 0, sum};
  //    d_communicator->send(D2H_DEBUG, 4*sizeof(uint),m);
  //  }

  //}
  //inline __device__ void EventManager::debugWorkItemCounterAdd(int num)
  //{
  //  atomicAdd((int*)&WorkItemCounter[num],1);
  //}
}
#endif
