/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*   Rostislav Khlebnikov /  khlebnikov ( at ) icg.tugraz.at
*/

#include "distributor/hscheduler.h"
#include "distributor/hscheduler.cuh"
#include "distributor/eventlist.h"
#include "tools/utils.h"

#include <algorithm>

namespace Softshell
{
  HScheduler::HScheduler(DevicePointerList* _devicepointerlist, ConfigCollection* _config) :
    devicepointerlist( _devicepointerlist ) , config(_config)
  {

  }

 std::vector<uint> HScheduler::initDevices(uint deviceId, const Flags& flags, const ControllerInfo* info)
  {
    std::vector<uint> dlist;
    if(deviceId != ANY_DEVICE)
      dlist.push_back(deviceId);
    else
    {
      getCudaDevices(dlist, flags);
    }
    if(dlist.size() == 0)
      throw(Softshell::error("No useable CUDA Devices found\n"));
    return initDevices(dlist, flags, info);
  }

  std::vector<uint> HScheduler::initDevices(std::vector<uint> const & devicelist, const Flags& flags,  const ControllerInfo* info)
  {
    std::vector<uint> inlist = devicelist;
    std::sort(inlist.begin(), inlist.end());
    std::vector<uint> reflist;
    getCudaDevices(reflist, flags);

    std::vector<uint> outlist;


    for(std::vector<uint>::const_iterator itin = inlist.begin(), itref = reflist.begin(); itin != inlist.end(); )
    {
      if(itref == reflist.end())
      {
        std::stringstream ss;
        ss << *itin << " is no useable device id";
        throw(Softshell::error(ss.str()));
      }
      if(*itref != *itin)
      {
        ++itref;
        continue;
      }

      outlist.push_back(*itin);
      Device* d = new Device(*itin, flags, config, info);
      EventList::getInstance()->addDevice(d);
      deviceToIdMapping.insert(std::make_pair(d, *itin));
      idToDeviceMapping.insert(std::make_pair(*itin, d));
      ++itin; ++itref;
    }

    return outlist;
  }

  void HScheduler::reset()
  {
    //printf("HScheduler::reset()\n");
    for(IDToDeviceMappingType::const_iterator it = idToDeviceMapping.begin(); it != idToDeviceMapping.end(); ++it)
    {
      //printf("Shutting down device %d \n", it->first);
      it->second->shutdown();
      delete it->second;
    }
    idToDeviceMapping.clear();
    deviceToIdMapping.clear();
    //printf("HScheduler::reset() finished\n");
    //TODO: stop all and cleanup
  }

}
