/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*   Rostislav Khlebnikov /  khlebnikov ( at ) icg.tugraz.at
*/

#include "cuda.h"
#include "api/scheduler.h"
#include "distributor/hscheduler.cuh"
#include "distributor/hscheduler.h"
#include "tools/utils.h"
#include "distributor/devicepointerlist.h"
#include "distributor/event_impl.h"
#include "megakernel/controller.h"

#include <iostream>
#include <algorithm>


namespace Softshell
{
  void getCudaDevices(std::vector<uint>& devicelist, const Flags& flags)
  {
    devicelist.clear();
    int count;
    CUDA_CHECKED_CALL(cudaGetDeviceCount( &count ) );
    for(int i = 0; i < count; ++i)
    {
      //TODO check flags
      cudaDeviceProp props;
      CUDA_CHECKED_CALL(cudaGetDeviceProperties(&props, i));
      if(props.major >= 2)
      {
        devicelist.push_back(i);
#ifdef _DEBUG
        std::cout << "Softshell: Found useable device  " << props.name << std::endl;
#endif
      }
      else
      {
#ifdef _DEBUG
        std::cout << "Softshell: Will not use " << props.name << " due to lacking Compute Capability" << std::endl;
#endif
      }
    }
  }





  Scheduler::Scheduler(int device_id, const Flags& flags) : hscheduler(0)
  {
    init(device_id, flags);
    //devicepointerlist = new DevicePointerList();
    //hscheduler = new HScheduler(devicepointerlist, &config);
    //deviceid_list = hscheduler->initDevices(device_id, flags);
  }
  Scheduler::Scheduler(std::vector<uint> const& device_ids, const Flags& flags) : hscheduler(0)
  {
    init(device_ids, flags);
    //devicepointerlist = new DevicePointerList();
    //hscheduler = new HScheduler(devicepointerlist, &config);
    //deviceid_list = hscheduler->initDevices(device_ids, flags);
  }
  Scheduler::Scheduler(const ControllerInfo& info, int device_id, const Flags& flags) : hscheduler(0)
  {
    init(info, device_id, flags);
  }
  Scheduler::Scheduler(const ControllerInfo& info, std::vector<uint> const & device_ids, const Flags& flags) : hscheduler(0)
  {
    init(info, device_ids, flags);
  }



  bool Scheduler::init(int device_id, const Flags& flags)
  {
    if(hscheduler == 0)
    {
      devicepointerlist = new DevicePointerList();
      hscheduler = new HScheduler(devicepointerlist, &config);
    }
    else
      hscheduler->reset();
    deviceid_list = hscheduler->initDevices(device_id, flags);
    return is_init();
  }
  bool Scheduler::init(std::vector<uint> const& device_ids, const Flags& flags)
  {
    if(hscheduler == 0)
    {
      devicepointerlist = new DevicePointerList();
      hscheduler = new HScheduler(devicepointerlist, &config);
    }
    else
      hscheduler->reset();
    deviceid_list = hscheduler->initDevices(device_ids, flags);
    return is_init();
  }
  bool Scheduler::init(const ControllerInfo& info, int device_id, const Flags& flags)
  {
    config.registerConfig("Controller", info);
    config.set("Controller", info);
    config.get<ControllerInfo>("Controller").setConfig(&config);
    if(hscheduler == 0)
    {
      devicepointerlist = new DevicePointerList();
      hscheduler = new HScheduler(devicepointerlist, &config);
    }
    else
      hscheduler->reset();
    deviceid_list = hscheduler->initDevices(device_id, flags, &info);
    return is_init();
  }
  bool Scheduler::init(const ControllerInfo& info, std::vector<uint> const & device_ids, const Flags& flags)
  {
    config.registerConfig("Controller", info);
    config.set("Controller", info);
    config.get<ControllerInfo>("Controller").setConfig(&config);
    if(hscheduler == 0)
    {
      devicepointerlist = new DevicePointerList();
      hscheduler = new HScheduler(devicepointerlist, &config);
    }
    else
      hscheduler->reset();
    deviceid_list = hscheduler->initDevices(device_ids, flags, &info);
    return is_init();
  }

  bool Scheduler::is_init()
  {
    return deviceid_list.size() > 0;
  }

  void Scheduler::pause()
  {
    //pause all devices
    for(HScheduler::IDToDeviceMappingType::const_iterator it = hscheduler->idToDeviceMapping.begin(); it != hscheduler->idToDeviceMapping.end() ; ++it)
      it->second->pause();
  }
  void Scheduler::start()
  {
    //start devices
    for(HScheduler::IDToDeviceMappingType::const_iterator it = hscheduler->idToDeviceMapping.begin(); it != hscheduler->idToDeviceMapping.end() ; ++it)
      it->second->start();
  }
  void Scheduler::shutdown()
  {
    hscheduler->reset();
    delete hscheduler;
    delete devicepointerlist;
    hscheduler = 0;
    devicepointerlist = 0;

  }

  void Scheduler::setController(const ControllerInfo& controllerInfo)
  {
    config.set("Controller", controllerInfo);
    config.get<ControllerInfo>("Controller").setConfig(&config);
  }

  Device* Scheduler::getDevice(int device_id)
  {
    return hscheduler->getDevice(device_id);
  }
}
