/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#ifndef SOFTSHELL_MEGAKERNEL_CONTROLLER_CUH_INCLUDED
#define SOFTSHELL_MEGAKERNEL_CONTROLLER_CUH_INCLUDED

#include "megakernel/controller.h"
#include "megakernel/launchstates.cuh"
#include "communicator/dcommunicator.cuh"
#include "queue/queue.cuh"
#include "tools/common.cuh"
#include "tools/utils.h"
#include "tools/config.h"
#include "distributor/proceduremanager.cuh"
#include "megakernel/execstate.cuh"
#include "aggregation/aggregation.cuh"
#include <map>



namespace Softshell
{
  extern __constant__ int stopOnNoWork;

  template<class PriorityEval>
  class PriorityCaller
  {
  public:
    __device__ static typename PriorityEval::PriorityType priority(Workpackage* wp, DeviceEntryProcedureFunc func)
    {
      return PriorityEval::eval(wp, d_procedureManager->get(func));
    }
  };

  template<class PriorityEval, bool SortQueue, bool AggregateWorkItems>
  __device__ void Controller(volatile BlockExecState * state, uint sortingThreads)
  {
    extern __shared__ uint myshared[];
    __shared__ bool hadWork;

    if(stopOnNoWork && threadIdx.x == 0)
     hadWork = *(volatile int*)(&d_queue.count) > 0;

    //__shared__ bool runmaintain;
    state->run = true;
    //runmaintain = true;
	  for(int i = 0; i < 1000; ++i)
    //while(true)
    {
      //check messages
      if(d_communicator->dispatch(threadIdx.x, blockDim.x) && threadIdx.x == 0)
        hadWork = 1;
      __syncthreads();

      //sort queue
      if(SortQueue)
      {
        d_queue.sortIteration<typename PriorityEval::PriorityType, PriorityCaller<PriorityEval> >(myshared, (typename PriorityEval::PriorityType*)(myshared+2*sortingThreads), 2*min(blockDim.x,sortingThreads), threadIdx.x, blockDim.x);
        __syncthreads();
      }

      //maintain thread aggregation
      if(AggregateWorkItems)
      {
        d_aggregation->maintain();
        __syncthreads();
      }

      if(stopOnNoWork && threadIdx.x == 0)
        if(hadWork && *(volatile int*)(&d_queue.count) == 0)
        {
          //printf("%d %d controller end due to no work\n", state->launchId, blockIdx.x);
          d_launchStates->endLaunch(state->launchId);
          state->run = false;
        }


      state->checkRunstate(threadIdx.x == 0);
      __syncthreads();
      if(!state->run)
        return;

      if(hadWork)
      {
        volatile int i = 0;
        for( ; i < 1000; ++i);
      }
      __syncthreads();
    }
    //printf("%d controller end due to runs\n",state->launchId);
  }

  template<class PriorityEval, bool SortQueue, bool AggregateWorkItems>
  __global__ void d_separateController(uint launchId, uint sortingThreads, uint additionalSharedMemory)
  {
    ___i_shared_all[0] = d_launchStates->checkRunState(launchId);
    syncthreads();
    if(!___i_shared_all[0])
      return;

    volatile BlockExecState* bstate = (BlockExecState*)(___i_shared_all + (additionalSharedMemory-sizeof(BlockExecState))/sizeof(uint));
    bstate->launchId = launchId;
    bstate->run = true;
    bstate->starttime = getTimeCycles();
    Controller<PriorityEval, SortQueue, AggregateWorkItems>(bstate, sortingThreads);
  }

  template<class PriorityEval, bool SortQueue, bool AggregateWorkItems>
  __global__ void d_getControllerFunc(ControllerFunc* out)
  {
    *out = &Controller<PriorityEval, SortQueue, AggregateWorkItems>;
  }

  template<class PriorityEval, bool SortQueue = true, bool AggregateWorkItems = true>
  class ControllerInfoImpl : public ControllerInfoBase
  {
    static uint id;
    uint* sortingThreads;
    ControllerFunc readControllerFunc()
    {
      ControllerFunc *d_func, func;
      CUDA_CHECKED_CALL(cudaMalloc(&d_func, sizeof(ControllerFunc)));
      d_getControllerFunc<PriorityEval, SortQueue, AggregateWorkItems><<<1,1>>>(d_func);
      CUDA_CHECKED_CALL(cudaMemcpy(&func, d_func, sizeof(ControllerFunc), cudaMemcpyDeviceToHost));
      CUDA_CHECKED_CALL(cudaFree(d_func));
      return func;
    }
  public:
    ControllerInfoImpl() :
        sortingThreads(0)
    {
      if(id == 0)
        id = getFreeId();
    }
    void setConfig(Config* config)
    {
      sortingThreads = &config->registerConfig("ControllerSortingThreads", 256U);
    }
    bool doesSorting() const
    {
      return SortQueue;
    }
    uint sortingKeySize() const
    {
      return SortQueue?sizeof(typename PriorityEval::PriorityType):0;
    }
     uint getId() const
    {
      return id;
    }
    uint requiredSharedMemory() const
    {
      if(!SortQueue)
        return 0;
      if(sortingThreads)
        return *sortingThreads*2*(sizeof(typename PriorityEval::PriorityType)+sizeof(uint));
      return 256*(sizeof(typename PriorityEval::PriorityType)+sizeof(uint));
    }
    ControllerFunc get()
    {
      typedef std::map<int, ControllerFunc> FunctsType;
      static FunctsType functs;
      int devid;
      cudaGetDevice(&devid);
      //have we got entry for that device already?
      FunctsType::iterator found = functs.find(devid);
      if(found == functs.end())
        found = functs.insert(std::make_pair(devid,readControllerFunc())).first;
      return found->second;
    }
    void launchController(uint launchId, cudaStream_t& stream)
    {
      uint smem = requiredSharedMemory() + 16U;
      uint threads = 128;
      uint sthreads = 0;

      if(sortingThreads)
      {
        sthreads = *sortingThreads;
        threads = std::max(*sortingThreads, 128u);
      }
      d_separateController<PriorityEval, SortQueue, AggregateWorkItems><<<1, threads, smem, stream>>>(launchId, sthreads, smem);
    }
    ControllerInfoBase* clone() const
    {
      return new ControllerInfoImpl(*this);
    }
  };
  template<class PriorityEval, bool SortQueue, bool AggregateWorkItems>
  uint ControllerInfoImpl<PriorityEval,SortQueue,AggregateWorkItems>::id = 0;

  template<class PriorityEval, bool SortQueue, bool AggregateWorkItems>
  ControllerInfo getController()
  {
    return ControllerInfo(ControllerInfoImpl<PriorityEval,SortQueue,AggregateWorkItems>());
  }

}

#endif //SOFTSHELL_MEGAKERNEL_CONTROLLER_CUH_INCLUDED
