/*
  Softshell: Dynamic Scheduling on GPUs.
  http://www.icg.tugraz.at/project/mvp

  Copyright (C) 2012 Institute for Computer Graphics and Vision,
                     Graz University of Technology

  Author(s):  Markus Steinberger - steinberger ( at ) icg.tugraz.at
              Bernhard Kainz - kainz ( at ) icg.tugraz.at
              Michael Kenzel - kenzel ( at ) icg.tugraz.at
              Stefan Hauswiesner - hauswiesner ( at ) icg.tugraz.at
              Bernhard Kerbl - kerbl ( at ) icg.tugraz.at
              Dieter Schmalstieg - schmalstieg ( at ) icg.tugraz.at

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/


/*
* file created by    Markus Steinberger / steinberger ( at ) icg.tugraz.at
*
* modifications by
*/

#ifndef SOFTSHELL_MEGAKERNEL_CONTROLLER_INCLUDED
#define SOFTSHELL_MEGAKERNEL_CONTROLLER_INCLUDED

#include "tools/types.h"

namespace Softshell
{
  class Config;
  struct BlockExecState;
  typedef void(*ControllerFunc)(volatile BlockExecState * state, uint sortingThreads);


  class ControllerInfoBase
  {
  protected:
    static uint getFreeId()
    {
      static uint next = 0;
      return ++next;
    }
  public:
    virtual bool doesSorting() const = 0;
    virtual uint sortingKeySize() const = 0;
    virtual uint getId() const = 0;
    virtual uint requiredSharedMemory() const = 0;
    virtual ControllerFunc get() = 0;
    virtual ControllerInfoBase* clone() const = 0;
    virtual void launchController(uint launchId, cudaStream_t& stream) = 0;
    virtual void setConfig(Config* config) = 0;
    virtual ~ControllerInfoBase() { }
  };

  class ControllerInfo
  {
    ControllerInfoBase* allDeviceInfo;
  public:

    ControllerFunc func()
    {
      return allDeviceInfo->get();
    }
    bool doesSorting() const
    {
      return allDeviceInfo->doesSorting();
    }
    uint sortingKeySize() const
    {
      return allDeviceInfo->sortingKeySize();
    }
    uint requiredSharedMemory() const
    {
      return allDeviceInfo->requiredSharedMemory();
    }
    void launchController(uint launchId, cudaStream_t& stream)
    {
      allDeviceInfo->launchController(launchId, stream);
    }
    void setConfig(Config* config)
    {
      allDeviceInfo->setConfig(config);
    }
    
    ControllerInfo() : allDeviceInfo(0) { }
    ControllerInfo(const ControllerInfoBase& info) : allDeviceInfo(info.clone()) 
    {
    }
    ControllerInfo(const ControllerInfo& other) : allDeviceInfo(other.allDeviceInfo->clone()) 
    {
    }
    const ControllerInfo& operator= (const ControllerInfoBase& other)
    {
      allDeviceInfo = other.clone();
      return *this;
    }
    ~ControllerInfo()
    {
      if(allDeviceInfo)
        delete allDeviceInfo;
    }
    bool operator== (const ControllerInfo& other)
    {
      return allDeviceInfo->getId() == other.allDeviceInfo->getId() && doesSorting() == other.doesSorting() && sortingKeySize() == other.sortingKeySize();
    }
  };
  
}

#endif //SOFTSHELL_MEGAKERNEL_CONTROLLER_INCLUDED